<?php
// wallet/process.php

// Enable error reporting for debugging (remove in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Start session if not already started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Verify the file path is correct (adjust as needed)
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/functions.php';

// Debugging check - remove after testing
if (!function_exists('isLoggedIn')) {
    die('Error: isLoggedIn() function not found. Check functions.php');
}

if (!isLoggedIn()) {
    header('HTTP/1.1 401 Unauthorized');
    die(json_encode(['error' => 'Unauthorized']));
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    header('HTTP/1.1 405 Method Not Allowed');
    die(json_encode(['error' => 'Method Not Allowed']));
}

// Validate amount
$amount = isset($_GET['amount']) ? floatval($_GET['amount']) : 0;
if ($amount <= 0) {
    header('HTTP/1.1 400 Bad Request');
    die(json_encode(['error' => 'Invalid amount']));
}

// Validate payment method
$payment_method = isset($_GET['payment_method']) ? sanitizeInput($_GET['payment_method']) : '';
$allowed_methods = ['esewa', 'khalti', 'imepay', 'fonepay'];
if (!in_array($payment_method, $allowed_methods)) {
    header('HTTP/1.1 400 Bad Request');
    die(json_encode(['error' => 'Invalid payment method']));
}

try {
    // Verify database connection
    if (!isset($pdo) || !($pdo instanceof PDO)) {
        throw new Exception('Database connection not established');
    }

    // Create transaction
    $reference = "Deposit via " . ucfirst($payment_method);
    
    // Verify addWalletTransaction exists
    if (!function_exists('addWalletTransaction')) {
        throw new Exception('addWalletTransaction function not found');
    }
    
    $transaction_id = addWalletTransaction(
        $_SESSION['user_id'],
        $amount,
        'deposit',
        $reference,
        'pending'
    );
    
    if (!$transaction_id) {
        throw new Exception('Failed to create transaction');
    }

    // Successful transaction creation
    header("Location: success.php?transaction_id=" . urlencode($transaction_id));
    exit();
    
} catch (PDOException $e) {
    error_log("Database Error: " . $e->getMessage());
    header('HTTP/1.1 500 Internal Server Error');
    die(json_encode(['error' => 'Database error occurred']));
} catch (Exception $e) {
    error_log("Transaction Error: " . $e->getMessage());
    header('HTTP/1.1 500 Internal Server Error');
    die(json_encode([
        'error' => 'Transaction processing failed',
        'details' => $e->getMessage() // Remove in production
    ]));
}