<?php
// wallet/index.php

// Use absolute paths
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/functions.php';

// Initialize session if not started
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Strict authentication check
if (!isLoggedIn()) {
    $_SESSION['redirect_url'] = $_SERVER['REQUEST_URI'];
    redirect('../login.php');
    exit;
}

// Ensure database connection
if (!isset($pdo)) {
    try {
        $pdo = getDBConnection();
    } catch (PDOException $e) {
        error_log("Wallet DB Error: " . $e->getMessage());
        die("System temporarily unavailable. Please try again later.");
    }
}

// Rest of your code...

$user = getUserById($_SESSION['user_id']);
$wallet_balance = getWalletBalance($_SESSION['user_id']);

// Get wallet transactions
$stmt = $pdo->prepare("
    SELECT wt.* 
    FROM wallet_transactions wt
    JOIN wallet w ON wt.wallet_id = w.id
    WHERE w.user_id = ?
    ORDER BY wt.created_at DESC
    LIMIT 50
");
$stmt->execute([$_SESSION['user_id']]);
$transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);

$pageTitle = "Wallet - " . SITE_NAME;
require_once 'header.php';
?>

<style>
    /* Wallet Specific Styles */
    .dashboard-container {
        display: flex;
        min-height: calc(100vh - 70px);
        margin-top: 70px;
    }
    
    .sidebar {
        width: 250px;
        background-color: #2c3e50;
        color: white;
        padding: 1.5rem 0;
        position: fixed;
        height: calc(100vh - 70px);
    }
    
    .user-profile {
        text-align: center;
        padding: 0 1rem 1.5rem;
        border-bottom: 1px solid rgba(255,255,255,0.1);
    }
    
    .avatar {
        font-size: 4rem;
        color: #bdc3c7;
        margin-bottom: 0.5rem;
    }
    
    .nav-links {
        margin-top: 1rem;
        list-style: none;
    }
    
    .nav-item {
        padding: 0.5rem 1.5rem;
    }
    
    .nav-link {
        color: #ecf0f1;
        text-decoration: none;
        display: flex;
        align-items: center;
        transition: all 0.3s;
    }
    
    .nav-link:hover, 
    .nav-link.active {
        color: var(--primary-color);
    }
    
    .nav-link i {
        width: 20px;
        margin-right: 0.5rem;
        text-align: center;
    }
    
    .main-content {
        flex: 1;
        margin-left: 250px;
        padding: 2rem;
    }
    
    .wallet-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        margin-bottom: 2rem;
    }
    
    .balance-card {
        background: white;
        border-radius: 8px;
        padding: 1.5rem;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        margin-bottom: 2rem;
    }
    
    .balance-info h3 {
        color: #7f8c8d;
        font-size: 1rem;
        margin-bottom: 0.5rem;
    }
    
    .balance-info h1 {
        font-size: 2.5rem;
        color: var(--text-color);
        margin: 0;
    }
    
    .btn {
        display: inline-flex;
        align-items: center;
        padding: 0.75rem 1.5rem;
        border-radius: 6px;
        font-weight: 500;
        text-decoration: none;
        transition: all 0.3s;
        border: none;
        cursor: pointer;
    }
    
    .btn i {
        margin-right: 0.5rem;
    }
    
    .btn-primary {
        background-color: var(--primary-color);
        color: white;
    }
    
    .btn-primary:hover {
        background-color: var(--accent-color);
    }
    
    .transactions-card {
        background: white;
        border-radius: 8px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        overflow: hidden;
    }
    
    .card-header {
        padding: 1rem 1.5rem;
        border-bottom: 1px solid #eee;
    }
    
    .card-header h3 {
        font-size: 1.2rem;
        color: var(--text-color);
        margin: 0;
    }
    
    .card-body {
        padding: 1.5rem;
    }
    
    .table {
        width: 100%;
        border-collapse: collapse;
    }
    
    .table th, .table td {
        padding: 0.75rem 1rem;
        text-align: left;
        border-bottom: 1px solid #eee;
    }
    
    .table th {
        font-weight: 600;
        color: var(--text-color);
    }
    
    .text-success {
        color: #2e7d32;
    }
    
    .text-danger {
        color: #c62828;
    }
    
    .text-warning {
        color: #ff8f00;
    }
    
    .text-muted {
        color: #7f8c8d;
    }
    
    .modal {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background-color: rgba(0,0,0,0.5);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 2000;
        opacity: 0;
        visibility: hidden;
        transition: all 0.3s;
    }
    
    .modal.active {
        opacity: 1;
        visibility: visible;
    }
    
    .modal-content {
        background: white;
        border-radius: 8px;
        width: 100%;
        max-width: 500px;
        box-shadow: 0 4px 20px rgba(0,0,0,0.15);
    }
    
    .modal-header {
        padding: 1rem 1.5rem;
        border-bottom: 1px solid #eee;
        display: flex;
        justify-content: space-between;
        align-items: center;
    }
    
    .modal-header h3 {
        font-size: 1.2rem;
        margin: 0;
    }
    
    .close-btn {
        background: none;
        border: none;
        font-size: 1.5rem;
        cursor: pointer;
        color: #7f8c8d;
    }
    
    .modal-body {
        padding: 1.5rem;
    }
    
    .form-group {
        margin-bottom: 1.5rem;
    }
    
    .form-label {
        display: block;
        margin-bottom: 0.5rem;
        font-weight: 500;
    }
    
    .form-control {
        width: 100%;
        padding: 0.75rem;
        border: 1px solid #d1d3e2;
        border-radius: 6px;
        font-size: 1rem;
    }
    
    .form-control:focus {
        border-color: var(--primary-color);
        outline: none;
        box-shadow: 0 0 0 2px rgba(78, 115, 223, 0.25);
    }
    
    .form-select {
        width: 100%;
        padding: 0.75rem;
        border: 1px solid #d1d3e2;
        border-radius: 6px;
        font-size: 1rem;
        background-color: white;
    }
    
    .qr-code-container {
        text-align: center;
        margin-bottom: 1.5rem;
        display: none;
    }
    
    .qr-code-container.active {
        display: block;
    }
    
    .qr-code-img {
        max-width: 200px;
        margin: 0 auto;
    }
    
    @media (max-width: 768px) {
        .dashboard-container {
            flex-direction: column;
        }
        
        .sidebar {
            width: 100%;
            position: static;
            height: auto;
        }
        
        .main-content {
            margin-left: 0;
            padding: 1rem;
        }
        
        .wallet-header {
            flex-direction: column;
            align-items: flex-start;
        }
        
        .balance-card {
            width: 100%;
        }
        
        .table {
            display: block;
            overflow-x: auto;
        }
    }
</style>

<div class="dashboard-container">
    <div class="sidebar">
        <div class="user-profile">
            <div class="avatar">
                <i class="fas fa-user-circle"></i>
            </div>
            <h3><?= htmlspecialchars($user['username'], ENT_QUOTES, 'UTF-8') ?></h3>
            <p><?= htmlspecialchars($user['email'], ENT_QUOTES, 'UTF-8') ?></p>
        </div>
        
        <ul class="nav-links">
            <li class="nav-item">
                <a href="../dashboard.php" class="nav-link">
                    <i class="fas fa-tachometer-alt"></i> Dashboard
                </a>
            </li>
            <li class="nav-item">
                <a href="../player/" class="nav-link">
                    <i class="fas fa-play-circle"></i> Watch TV
                </a>
            </li>
            <li class="nav-item">
                <a href="../subscriptions.php" class="nav-link">
                    <i class="fas fa-tv"></i> Subscriptions
                </a>
            </li>
            <li class="nav-item">
                <a href="#" class="nav-link active">
                    <i class="fas fa-wallet"></i> Wallet
                </a>
            </li>
            <li class="nav-item">
                <a href="../logout.php" class="nav-link">
                    <i class="fas fa-sign-out-alt"></i> Logout
                </a>
            </li>
        </ul>
    </div>
    
    <div class="main-content">
        <div class="wallet-header">
            <h2>Wallet</h2>
        </div>
        
        <div class="balance-card">
            <div class="balance-info">
                <h3>Current Balance</h3>
                <h1>Nrs. <?= number_format($wallet_balance, 2) ?></h1>
            </div>
            <button id="openAddFunds" class="btn btn-primary">
                <i class="fas fa-plus-circle"></i> Add Funds
            </button>
        </div>
        
        <div class="transactions-card">
            <div class="card-header">
                <h3>Transaction History</h3>
            </div>
            <div class="card-body">
                <?php if (empty($transactions)): ?>
                    <p class="text-muted">No transactions yet.</p>
                <?php else: ?>
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Date</th>
                                    <th>Type</th>
                                    <th>Amount</th>
                                    <th>Status</th>
                                    <th>Reference</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($transactions as $transaction): 
                                    $type_class = $transaction['transaction_type'] === 'deposit' ? 'text-success' : 'text-danger';
                                    $status_class = $transaction['status'] === 'completed' ? 'text-success' : 
                                                  ($transaction['status'] === 'failed' ? 'text-danger' : 'text-warning');
                                ?>
                                <tr>
                                    <td><?= date('M d, Y H:i', strtotime($transaction['created_at'])) ?></td>
                                    <td><?= ucfirst($transaction['transaction_type']) ?></td>
                                    <td class="<?= $type_class ?>">
                                        <?= $transaction['transaction_type'] === 'deposit' ? '+' : '-' ?>
                                        Nrs. <?= number_format($transaction['amount'], 2) ?>
                                    </td>
                                    <td class="<?= $status_class ?>">
                                        <?= ucfirst($transaction['status']) ?>
                                    </td>
                                    <td><?= $transaction['reference'] ? htmlspecialchars($transaction['reference']) : 'N/A' ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Add Funds Modal -->
<div class="modal" id="addFundsModal">
    <div class="modal-content">
        <div class="modal-header">
            <h3>Add Funds to Wallet</h3>
            <button class="close-btn" id="closeModal">&times;</button>
        </div>
        <div class="modal-body">
            <form id="addFundsForm" action="process.php" method="GET">
                <div class="form-group">
                    <label for="amount" class="form-label">Amount (Nrs.)</label>
                    <input type="number" class="form-control" id="amount" name="amount" min="1" step="0.01" required>
                </div>
                <div class="form-group">
                    <label for="payment_method" class="form-label">Payment Method</label>
                    <select class="form-select" id="payment_method" name="payment_method" required>
                        <option value="">Select Payment Method</option>
                        <option value="esewa">eSewa</option>
                        <option value="khalti">Khalti</option>
                        <option value="imepay">IME Pay</option>
                        <option value="fonepay">Fonepay (QR)</option>
                    </select>
                </div>
                <div class="qr-code-container" id="qrCodeContainer">
                    <p class="text-muted">Scan the QR code to pay</p>
                    <img src="../assets/img/fonepay-qr.png" alt="Fonepay QR" class="qr-code-img">
                    <p class="text-muted">Amount: Nrs. <span id="qrAmount">0.00</span></p>
                </div>
                <button type="submit" class="btn btn-primary w-100">Proceed to Payment</button>
            </form>
        </div>
    </div>
</div>

<script>
// Modal handling
const modal = document.getElementById('addFundsModal');
const openBtn = document.getElementById('openAddFunds');
const closeBtn = document.getElementById('closeModal');

openBtn.addEventListener('click', () => {
    modal.classList.add('active');
});

closeBtn.addEventListener('click', () => {
    modal.classList.remove('active');
});

// Payment method change handler
document.getElementById('payment_method').addEventListener('change', function() {
    const qrContainer = document.getElementById('qrCodeContainer');
    if (this.value === 'fonepay') {
        qrContainer.classList.add('active');
        document.getElementById('qrAmount').textContent = document.getElementById('amount').value || '0.00';
    } else {
        qrContainer.classList.remove('active');
    }
});

// Amount input handler
document.getElementById('amount').addEventListener('input', function() {
    if (document.getElementById('payment_method').value === 'fonepay') {
        document.getElementById('qrAmount').textContent = this.value || '0.00';
    }
});

// Close modal when clicking outside
window.addEventListener('click', (e) => {
    if (e.target === modal) {
        modal.classList.remove('active');
    }
});
</script>

<?php require_once 'footer.php'; ?>