class SubscriptionHandler {
    constructor() {
        this.initEventListeners();
    }

    initEventListeners() {
        document.querySelectorAll('.btn-purchase-subscription').forEach(button => {
            button.addEventListener('click', this.handlePurchase.bind(this));
        });
    }

    async handlePurchase(event) {
        const button = event.currentTarget;
        const card = button.closest('.subscription-card');
        const subscriptionId = card?.dataset?.subscriptionId;

        // Validate subscription ID
        if (!subscriptionId || !/^\d+$/.test(subscriptionId)) {
            this.showError('Invalid subscription selected');
            return;
        }

        this.setButtonState(button, true);

        try {
            const response = await this.sendPurchaseRequest(subscriptionId);
            
            if (response.success) {
                this.showSuccess(response.message);
                this.updateUI(response.data);
            } else {
                this.handlePurchaseError(response);
            }
        } catch (error) {
            console.error('Purchase error:', error);
            this.showError(error.message || 'Failed to process purchase');
        } finally {
            this.setButtonState(button, false);
        }
    }

    async sendPurchaseRequest(subscriptionId) {
        const response = await fetch('/IPTV/root/api/purchase_subscription.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            },
            body: JSON.stringify({ 
                subscription_id: parseInt(subscriptionId)
            }),
            credentials: 'same-origin'
        });

        if (!response.ok) {
            const errorData = await response.json();
            throw new Error(errorData.message || `HTTP error! status: ${response.status}`);
        }

        return await response.json();
    }

    handlePurchaseError(response) {
        if (response.details?.includes('Insufficient funds')) {
            this.showFundsModal(
                parseFloat(response.required_amount || 0),
                parseFloat(response.current_balance || 0)
            );
        } else {
            this.showError(response.message || 'Purchase failed');
        }
    }

    setButtonState(button, isLoading) {
        button.disabled = isLoading;
        button.innerHTML = isLoading
            ? '<span class="spinner-border spinner-border-sm"></span> Processing...'
            : button.dataset.originalText || 'Purchase Now';
    }

    showError(message) {
        const toastEl = document.getElementById('errorToast');
        const toastBody = toastEl.querySelector('.toast-body');
        toastBody.textContent = message;
        new bootstrap.Toast(toastEl).show();
    }

    showSuccess(message) {
        const toastEl = document.getElementById('successToast');
        const toastBody = toastEl.querySelector('.toast-body');
        toastBody.textContent = message;
        new bootstrap.Toast(toastEl).show();
    }

    showFundsModal(requiredAmount, currentBalance) {
        const neededAmount = (parseFloat(requiredAmount) - parseFloat(currentBalance)).toFixed(2);
        document.getElementById('requiredAmount').textContent = neededAmount;
        new bootstrap.Modal('#fundsModal').show();
    }

    updateUI(data) {
        // Safely update wallet balance
        const balanceElement = document.getElementById('walletBalance');
        if (balanceElement) {
            const balance = parseFloat(data?.balance || data?.new_balance || 0);
            balanceElement.textContent = balance.toFixed(2);
        }

        // Update current subscription
        const subElement = document.getElementById('currentSubscription');
        if (subElement && data?.subscription) {
            subElement.textContent = data.subscription.name;
            const expiryElement = document.getElementById('subscriptionExpiry');
            if (expiryElement && data.subscription.end_date) {
                expiryElement.textContent = new Date(data.subscription.end_date).toLocaleDateString();
            }
        }

        // Refresh subscription buttons
        document.querySelectorAll('.subscription-card').forEach(card => {
            const btn = card.querySelector('.btn-purchase-subscription');
            if (btn && card.dataset.subscriptionId == data?.subscription?.id) {
                btn.disabled = true;
                btn.classList.remove('btn-primary');
                btn.classList.add('btn-success');
                btn.innerHTML = '<i class="bi bi-check-circle"></i> Current Plan';
            }
        });
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    // Store original button text
    document.querySelectorAll('.btn-purchase-subscription').forEach(btn => {
        btn.dataset.originalText = btn.textContent;
    });
    
    new SubscriptionHandler();
});