<?php
require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/includes/functions.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

// Set currency to NPR
define('CURRENCY', 'Nrs.');

$user = getUserById($_SESSION['user_id']);
$wallet_balance = getWalletBalance($_SESSION['user_id']);
$current_subscription = getUserSubscription($_SESSION['user_id']);
$all_subscriptions = getAllSubscriptions();

$pageTitle = "Subscriptions - " . SITE_NAME;
require_once __DIR__ . '/includes/header.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle, ENT_QUOTES, 'UTF-8') ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #4e73df;
            --secondary-color: #f8f9fc;
            --accent-color: #2e59d9;
            --text-color: #5a5c69;
            --light-text: #ffffff;
            --success-color: #2e7d32;
            --danger-color: #c62828;
            --warning-color: #ff8f00;
            --border-radius: 0.35rem;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Nunito', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 
                        'Helvetica Neue', Arial, sans-serif;
            background-color: var(--secondary-color);
            color: var(--text-color);
            min-height: 100vh;
        }
        
        .container {
            width: 100%;
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        .dashboard-container {
            display: flex;
            min-height: calc(100vh - 70px);
            margin-top: 70px;
        }
        
        .sidebar {
            width: 250px;
            background-color: #2c3e50;
            color: white;
            padding: 1.5rem 0;
            position: fixed;
            height: calc(100vh - 70px);
        }
        
        .user-profile {
            text-align: center;
            padding: 0 1rem 1.5rem;
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }
        
        .avatar {
            font-size: 4rem;
            color: #bdc3c7;
            margin-bottom: 0.5rem;
        }
        
        .nav-links {
            margin-top: 1rem;
            list-style: none;
        }
        
        .nav-item {
            padding: 0.5rem 1.5rem;
        }
        
        .nav-link {
            color: #ecf0f1;
            text-decoration: none;
            display: flex;
            align-items: center;
            transition: all 0.3s;
        }
        
        .nav-link:hover, 
        .nav-link.active {
            color: var(--primary-color);
        }
        
        .nav-link i {
            width: 20px;
            margin-right: 0.5rem;
            text-align: center;
        }
        
        .main-content {
            flex: 1;
            margin-left: 250px;
            padding: 2rem;
        }
        
        .card {
            background: white;
            border-radius: var(--border-radius);
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
            margin-bottom: 1.5rem;
            overflow: hidden;
        }
        
        .card-header {
            padding: 1rem 1.5rem;
            background-color: var(--primary-color);
            color: white;
        }
        
        .card-header h5 {
            margin: 0;
        }
        
        .card-body {
            padding: 1.5rem;
        }
        
        .alert {
            padding: 1rem;
            border-radius: var(--border-radius);
            margin-bottom: 1.5rem;
        }
        
        .alert-info {
            background-color: #e7f5fe;
            color: #0c63e4;
        }
        
        .row {
            display: flex;
            flex-wrap: wrap;
            margin: 0 -0.75rem;
        }
        
        .col-lg-3 {
            flex: 0 0 25%;
            max-width: 25%;
            padding: 0 0.75rem;
        }
        
        .col-lg-9 {
            flex: 0 0 75%;
            max-width: 75%;
            padding: 0 0.75rem;
        }
        
        .col-md-6 {
            flex: 0 0 50%;
            max-width: 50%;
            padding: 0 0.75rem;
        }
        
        .col-lg-4 {
            flex: 0 0 33.333%;
            max-width: 33.333%;
            padding: 0 0.75rem;
        }
        
        .mb-0 {
            margin-bottom: 0 !important;
        }
        
        .mb-2 {
            margin-bottom: 0.5rem !important;
        }
        
        .mb-3 {
            margin-bottom: 1rem !important;
        }
        
        .mb-4 {
            margin-bottom: 1.5rem !important;
        }
        
        .mt-2 {
            margin-top: 0.5rem !important;
        }
        
        .mt-4 {
            margin-top: 1.5rem !important;
        }
        
        .py-4 {
            padding-top: 1.5rem !important;
            padding-bottom: 1.5rem !important;
        }
        
        .d-flex {
            display: flex !important;
        }
        
        .justify-content-between {
            justify-content: space-between !important;
        }
        
        .align-items-center {
            align-items: center !important;
        }
        
        .text-center {
            text-align: center !important;
        }
        
        .text-end {
            text-align: right !important;
        }
        
        .text-muted {
            color: #6c757d !important;
        }
        
        .text-primary {
            color: var(--primary-color) !important;
        }
        
        .text-success {
            color: var(--success-color) !important;
        }
        
        .text-danger {
            color: var(--danger-color) !important;
        }
        
        .text-warning {
            color: var(--warning-color) !important;
        }
        
        .bg-primary {
            background-color: var(--primary-color) !important;
        }
        
        .bg-success {
            background-color: var(--success-color) !important;
        }
        
        .bg-warning {
            background-color: var(--warning-color) !important;
        }
        
        .btn {
            display: inline-flex;
            align-items: center;
            padding: 0.75rem 1.5rem;
            border-radius: var(--border-radius);
            font-weight: 500;
            text-decoration: none;
            transition: all 0.3s;
            border: none;
            cursor: pointer;
        }
        
        .btn i {
            margin-right: 0.5rem;
        }
        
        .btn-primary {
            background-color: var(--primary-color);
            color: white;
        }
        
        .btn-primary:hover {
            background-color: var(--accent-color);
        }
        
        .btn-outline-primary {
            background-color: transparent;
            color: var(--primary-color);
            border: 1px solid var(--primary-color);
        }
        
        .btn-outline-primary:hover {
            background-color: var(--primary-color);
            color: white;
        }
        
        .btn-success {
            background-color: var(--success-color);
            color: white;
        }
        
        .btn-secondary {
            background-color: #6c757d;
            color: white;
        }
        
        .btn-secondary:hover {
            background-color: #5a6268;
        }
        
        .w-100 {
            width: 100% !important;
        }
        
        .flex-grow-1 {
            flex-grow: 1 !important;
        }
        
        .subscription-card {
            height: 100%;
            display: flex;
            flex-direction: column;
        }
        
        .subscription-card .card-body {
            flex-grow: 1;
            display: flex;
            flex-direction: column;
        }
        
        .subscription-card .card-text {
            margin-bottom: 1rem;
        }
        
        .toast-container {
            position: fixed;
            bottom: 20px;
            right: 20px;
            z-index: 1100;
        }
        
        .toast {
            display: flex;
            align-items: center;
            padding: 1rem;
            border-radius: var(--border-radius);
            color: white;
            margin-bottom: 1rem;
            opacity: 0;
            transition: opacity 0.3s;
            max-width: 350px;
        }
        
        .toast.show {
            opacity: 1;
        }
        
        .toast-success {
            background-color: var(--success-color);
        }
        
        .toast-error {
            background-color: var(--danger-color);
        }
        
        .toast-body {
            flex-grow: 1;
        }
        
        .close-btn {
            background: none;
            border: none;
            color: white;
            font-size: 1.25rem;
            cursor: pointer;
            margin-left: 1rem;
        }
        
        .modal {
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
            z-index: 2000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s;
        }
        
        .modal.show {
            opacity: 1;
            visibility: visible;
        }
        
        .modal-content {
            background: white;
            border-radius: var(--border-radius);
            width: 100%;
            max-width: 500px;
            box-shadow: 0 0.5rem 1rem rgba(0,0,0,0.15);
        }
        
        .modal-header {
            padding: 1rem 1.5rem;
            border-bottom: 1px solid #eee;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-header h5 {
            margin: 0;
        }
        
        .modal-body {
            padding: 1.5rem;
        }
        
        .modal-footer {
            padding: 1rem 1.5rem;
            border-top: 1px solid #eee;
            display: flex;
            justify-content: flex-end;
        }
        
        .spinner {
            display: inline-block;
            width: 1rem;
            height: 1rem;
            border: 2px solid rgba(255,255,255,0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        @media (max-width: 992px) {
            .col-lg-3,
            .col-lg-9 {
                flex: 0 0 100%;
                max-width: 100%;
            }
            
            .dashboard-container {
                flex-direction: column;
            }
            
            .sidebar {
                width: 100%;
                position: static;
                height: auto;
            }
            
            .main-content {
                margin-left: 0;
                padding: 1rem;
            }
        }
        
        @media (max-width: 768px) {
            .col-md-6,
            .col-lg-4 {
                flex: 0 0 100%;
                max-width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="dashboard-container">
        <div class="sidebar">
            <div class="user-profile">
                <div class="avatar">
                    <i class="fas fa-user-circle"></i>
                </div>
                <h3><?= htmlspecialchars($user['username'], ENT_QUOTES, 'UTF-8') ?></h3>
                <p><?= htmlspecialchars($user['email'], ENT_QUOTES, 'UTF-8') ?></p>
            </div>
            
            <ul class="nav-links">
                <li class="nav-item">
                    <a href="dashboard.php" class="nav-link">
                        <i class="fas fa-tachometer-alt"></i> Dashboard
                    </a>
                </li>
                <li class="nav-item">
                    <a href="player/" class="nav-link">
                        <i class="fas fa-play-circle"></i> Watch TV
                    </a>
                </li>
                <li class="nav-item">
                    <a href="subscriptions.php" class="nav-link active">
                        <i class="fas fa-tv"></i> Subscriptions
                    </a>
                </li>
                <li class="nav-item">
                    <a href="wallet/" class="nav-link">
                        <i class="fas fa-wallet"></i> Wallet
                    </a>
                </li>
                <li class="nav-item">
                    <a href="logout.php" class="nav-link">
                        <i class="fas fa-sign-out-alt"></i> Logout
                    </a>
                </li>
            </ul>
        </div>
        
        <div class="main-content">
            <div class="container py-4">
                <!-- Toast Notifications -->
                <div class="toast-container">
                    <div class="toast toast-error" id="errorToast">
                        <div class="toast-body" id="errorToastBody"></div>
                        <button type="button" class="close-btn">&times;</button>
                    </div>
                    
                    <div class="toast toast-success" id="successToast">
                        <div class="toast-body" id="successToastBody"></div>
                        <button type="button" class="close-btn">&times;</button>
                    </div>
                </div>

                <div class="row">
                    <div class="col-lg-3">
                        <div class="card mb-4">
                            <div class="card-header">
                                <h5 class="mb-0">Account Summary</h5>
                            </div>
                            <div class="card-body">
                                <div class="d-flex justify-content-between mb-3">
                                    <span>Wallet Balance:</span>
                                    <strong id="walletBalance"><?= CURRENCY ?> <?= number_format($wallet_balance, 2) ?></strong>
                                </div>
                                <div class="d-flex justify-content-between">
                                    <span>Current Plan:</span>
                                    <strong id="currentSubscription">
                                        <?= $current_subscription ? htmlspecialchars($current_subscription['name']) : 'None' ?>
                                    </strong>
                                </div>
                                <?php if ($current_subscription): ?>
                                <div class="mt-2 text-end small text-muted">
                                    Expires: <?= date('M d, Y', strtotime($current_subscription['end_date'])) ?>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Quick Actions</h5>
                            </div>
                            <div class="card-body">
                                <a href="wallet/" class="btn btn-outline-primary w-100 mb-2">
                                    <i class="fas fa-wallet"></i> Add Funds
                                </a>
                                <a href="player/" class="btn btn-outline-primary w-100">
                                    <i class="fas fa-play-circle"></i> Watch Now
                                </a>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-lg-9">
                        <div class="card">
                            <div class="card-header">
                                <h5 class="mb-0">Available Subscriptions</h5>
                            </div>
                            <div class="card-body">
                                <?php if (empty($all_subscriptions)): ?>
                                    <div class="alert alert-info">No subscriptions available at this time.</div>
                                <?php else: ?>
                                    <div class="row g-4">
                                        <?php foreach ($all_subscriptions as $sub): 
                                            $duration = $sub['duration_days'] >= 30 
                                                ? floor($sub['duration_days']/30) . ' Months' 
                                                : $sub['duration_days'] . ' Days';
                                            $is_current = $current_subscription && $current_subscription['subscription_id'] == $sub['id'];
                                            $can_purchase = !$is_current && ($wallet_balance >= $sub['price']);
                                        ?>
                                        <div class="col-md-6 col-lg-4">
                                            <div class="card subscription-card h-100" data-subscription-id="<?= $sub['id'] ?>">
                                                <div class="card-header <?= $is_current ? 'bg-success' : 'bg-primary' ?>">
                                                    <h5 class="card-title mb-0"><?= htmlspecialchars($sub['name']) ?></h5>
                                                </div>
                                                <div class="card-body d-flex flex-column">
                                                    <div class="mb-3 text-center">
                                                        <h3 class="<?= $is_current ? 'text-success' : 'text-primary' ?>">
                                                            <?= CURRENCY ?> <?= number_format($sub['price'], 2) ?>
                                                        </h3>
                                                        <small class="text-muted">per <?= $duration ?></small>
                                                    </div>
                                                    <p class="card-text flex-grow-1"><?= htmlspecialchars($sub['description']) ?></p>
                                                    
                                                    <?php if ($is_current): ?>
                                                        <button class="btn btn-success w-100" disabled>
                                                            <i class="fas fa-check-circle"></i> Current Plan
                                                        </button>
                                                    <?php else: ?>
                                                        <button class="btn btn-primary w-100 btn-purchase-subscription"
                                                            <?= !$can_purchase ? 'data-need-funds="true"' : '' ?>
                                                            <?= $current_subscription ? 'data-has-subscription="true"' : '' ?>>
                                                            <?= $current_subscription ? 'Already Subscribed' : ($can_purchase ? 'Purchase Now' : 'Add Funds to Purchase') ?>
                                                        </button>
                                                    <?php endif; ?>
                                                </div>
                                            </div>
                                        </div>
                                        <?php endforeach; ?>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Funds Modal -->
                <div class="modal" id="fundsModal">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">Additional Funds Required</h5>
                            <button type="button" class="close-btn">&times;</button>
                        </div>
                        <div class="modal-body">
                            <p>You need an additional <strong><?= CURRENCY ?> <span id="requiredAmount">0.00</span></strong> to purchase this subscription.</p>
                            <div class="d-flex justify-content-between mt-4">
                                <button type="button" class="btn btn-secondary" id="cancelFundsModal">Cancel</button>
                                <a href="wallet/" class="btn btn-primary">
                                    <i class="fas fa-wallet"></i> Add Funds
                                </a>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Existing Subscription Modal -->
                <div class="modal" id="existingSubscriptionModal">
                    <div class="modal-content">
                        <div class="modal-header bg-warning">
                            <h5 class="modal-title">Existing Subscription</h5>
                            <button type="button" class="close-btn">&times;</button>
                        </div>
                        <div class="modal-body">
                            <p>You already have an active subscription. Please wait until your current plan expires before purchasing a new one.</p>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-primary" id="closeExistingModal">OK</button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Toast handling
        const toasts = {
            error: {
                element: document.getElementById('errorToast'),
                body: document.getElementById('errorToastBody')
            },
            success: {
                element: document.getElementById('successToast'),
                body: document.getElementById('successToastBody')
            }
        };

        // Close buttons for toasts
        document.querySelectorAll('#errorToast .close-btn, #successToast .close-btn').forEach(btn => {
            btn.addEventListener('click', function() {
                this.closest('.toast').classList.remove('show');
            });
        });

        // Modal handling
        const modals = {
            funds: {
                element: document.getElementById('fundsModal'),
                cancel: document.getElementById('cancelFundsModal')
            },
            existing: {
                element: document.getElementById('existingSubscriptionModal'),
                close: document.getElementById('closeExistingModal')
            }
        };

        // Show modal function
        function showModal(modal) {
            modal.element.classList.add('show');
        }

        // Hide modal function
        function hideModal(modal) {
            modal.element.classList.remove('show');
        }

        // Modal event listeners
        modals.funds.cancel.addEventListener('click', () => hideModal(modals.funds));
        modals.existing.close.addEventListener('click', () => hideModal(modals.existing));

        // Show toast function
        function showToast(type, message) {
            const toast = toasts[type];
            toast.body.textContent = message;
            toast.element.classList.add('show');
            
            setTimeout(() => {
                toast.element.classList.remove('show');
            }, 5000);
        }

        // Handle subscription purchases
        document.querySelectorAll('.btn-purchase-subscription').forEach(button => {
            button.addEventListener('click', async function(e) {
                // Prevent purchase if already subscribed
                if (this.dataset.hasSubscription === 'true') {
                    e.preventDefault();
                    showModal(modals.existing);
                    return;
                }

                const card = this.closest('.subscription-card');
                const subscriptionId = card.dataset.subscriptionId;
                
                // Validate subscription ID
                if (!subscriptionId || isNaN(subscriptionId)) {
                    showToast('error', 'Invalid subscription selected');
                    return;
                }

                // Set loading state
                const originalText = this.innerHTML;
                this.disabled = true;
                this.innerHTML = '<span class="spinner"></span> Processing...';

                try {
                    const response = await fetch('../root/api/purchase_subscription.php', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            subscription_id: parseInt(subscriptionId)
                        })
                    });

                    const result = await response.json();

                    if (!response.ok) {
                        throw new Error(result.message || 'Purchase failed');
                    }

                    if (result.success) {
                        // Show success message
                        showToast('success', result.message);
                        
                        // Update UI
                        if (result.data) {
                            document.getElementById('walletBalance').textContent = '<?= CURRENCY ?> ' + result.data.balance.toFixed(2);
                            if (result.data.subscription) {
                                document.getElementById('currentSubscription').textContent = result.data.subscription.name;
                                // Refresh the page to update all subscription cards
                                setTimeout(() => location.reload(), 1500);
                            }
                        }
                    } else {
                        // Show error message
                        showToast('error', result.message);
                        
                        // Show funds modal if insufficient balance
                        if (result.required_amount) {
                            document.getElementById('requiredAmount').textContent = 
                                (result.required_amount - (result.current_balance || 0)).toFixed(2);
                            showModal(modals.funds);
                        }
                    }
                } catch (error) {
                    console.error('Purchase error:', error);
                    showToast('error', error.message);
                } finally {
                    // Reset button state
                    this.disabled = false;
                    this.innerHTML = originalText;
                }
            });
        });
    });
    </script>
</body>
</html>