document.addEventListener('DOMContentLoaded', function() {
    const player = document.getElementById('iptvPlayer');
    const channelItems = document.querySelectorAll('.channel-item');
    const channelSearch = document.getElementById('channelSearch');
    let hls = null;
    let retryCount = 0;
    const MAX_RETRIES = 3;
    
    // Check if URL is HLS
    function isHlsUrl(url) {
        return url.includes('.m3u8') || url.includes('hls');
    }
    
    // Play first channel by default
    if (channelItems.length > 0) {
        playChannel(channelItems[0]);
        channelItems[0].classList.add('active');
    }
    
    // Handle channel selection
    channelItems.forEach(item => {
        item.addEventListener('click', function() {
            showLoading();
            channelItems.forEach(i => i.classList.remove('active'));
            this.classList.add('active');
            playChannel(this);
        });
    });
    
    // Handle channel search
    channelSearch.addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase();
        channelItems.forEach(item => {
            const channelName = item.querySelector('.channel-name').textContent.toLowerCase();
            item.style.display = channelName.includes(searchTerm) ? 'flex' : 'none';
        });
    });
    
    // Main playback function
    function playChannel(channelElement) {
        retryCount = 0;
        const streamUrl = channelElement.dataset.url;
        
        // Destroy previous HLS instance if exists
        if (hls) {
            hls.destroy();
            hls = null;
        }
        
        // Reset player
        player.src = '';
        player.load();
        
        // For non-HLS streams (direct MP4, etc.)
        if (!isHlsUrl(streamUrl)) {
            player.src = streamUrl;
            player.load();
            player.play().then(() => {
                hideLoading();
            }).catch(e => {
                console.error('Playback failed:', e);
                showCustomAlert('Error playing this channel. Please try another one.');
                hideLoading();
            });
            return;
        }
        
        // First try native HLS for Safari/iOS
        if (player.canPlayType('application/vnd.apple.mpegurl')) {
            player.src = streamUrl;
            player.load();
            
            player.play().then(() => {
                hideLoading();
            }).catch(e => {
                console.error('Native HLS playback failed:', e);
                tryHlsJsPlayback(streamUrl);
            });
            return;
        }
        
        // Then try HLS.js for other browsers
        if (Hls.isSupported()) {
            tryHlsJsPlayback(streamUrl);
            return;
        }
        
        // Final fallback
        showCustomAlert('Your browser doesn\'t support HLS streaming. Try Chrome/Firefox or use VLC player.');
        hideLoading();
    }
    
    function tryHlsJsPlayback(streamUrl) {
        hls = new Hls({
            maxMaxBufferLength: 30,
            maxBufferSize: 60*1000*1000,
            maxBufferLength: 30,
            enableWorker: true,
            lowLatencyMode: true,
            backBufferLength: 30
        });
        
        hls.loadSource(streamUrl);
        hls.attachMedia(player);
        
        hls.on(Hls.Events.MANIFEST_PARSED, function() {
            player.play().then(() => {
                hideLoading();
            }).catch(e => {
                console.error('Auto-play failed:', e);
                player.muted = true;
                player.play().then(() => {
                    hideLoading();
                    player.muted = false;
                }).catch(e => {
                    hideLoading();
                    showCustomAlert('Please click the play button to start playback');
                });
            });
        });
        
        hls.on(Hls.Events.ERROR, function(event, data) {
            if (data.fatal) {
                switch(data.type) {
                    case Hls.ErrorTypes.NETWORK_ERROR:
                        if (retryCount < MAX_RETRIES) {
                            retryCount++;
                            showCustomAlert(`Network error. Retrying (${retryCount}/${MAX_RETRIES})...`, 'warning');
                            setTimeout(() => hls.startLoad(), 2000 * retryCount);
                        } else {
                            hideLoading();
                            showCustomAlert('Failed to connect after multiple attempts. Please try again later.', 'danger');
                        }
                        break;
                    case Hls.ErrorTypes.MEDIA_ERROR:
                        showCustomAlert('Media error. Trying to recover...', 'warning');
                        hls.recoverMediaError();
                        break;
                    default:
                        hideLoading();
                        showCustomAlert('Unable to play this stream. Try another channel.', 'danger');
                        hls.destroy();
                }
            }
        });
    }
    
    // Handle player errors
    player.addEventListener('error', function() {
        hideLoading();
        showCustomAlert('Error playing this channel. Please try another one.');
    });
    
    // Utility functions
    function showCustomAlert(message, type = 'danger') {
        const alert = document.getElementById('customAlert');
        const alertMessage = document.getElementById('alertMessage');
        
        alert.className = `alert alert-${type} position-fixed top-0 start-50 translate-middle-x mt-3`;
        alertMessage.textContent = message;
        alert.style.display = 'block';
        
        setTimeout(hideCustomAlert, 5000);
    }
    
    function hideCustomAlert() {
        document.getElementById('customAlert').style.display = 'none';
    }
    
    function showLoading() {
        document.getElementById('loadingSpinner').style.display = 'block';
    }
    
    function hideLoading() {
        document.getElementById('loadingSpinner').style.display = 'none';
    }
});