<?php
require_once '../includes/config.php';

if (!isLoggedIn()) {
    redirect('../login.php');
}

$user = getUserById($_SESSION['user_id']);
$subscription = getUserSubscription($_SESSION['user_id']);

if (!$subscription) {
    $_SESSION['error'] = "You need an active subscription to access the player";
    redirect('../subscriptions.php');
}

$channels = parseM3UPlaylist(IPTV_PLAYLIST_URL);

$pageTitle = "IPTV Player - " . SITE_NAME;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $pageTitle ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <style>
        :root {
            --primary-color: #e50914;
            --dark-bg: #0f0f0f;
            --light-text: #ffffff;
            --dark-text: #aaaaaa;
        }
        
        body {
            background-color: var(--dark-bg);
            color: var(--light-text);
            font-family: 'Helvetica Neue', Helvetica, Arial, sans-serif;
            min-height: 100vh;
        }
        
        .video-container {
            position: relative;
            padding-bottom: 56.25%; /* 16:9 aspect ratio */
            height: 0;
            overflow: hidden;
            background-color: #000;
        }
        
        .video-container video {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
        }
        
        .channel-list {
            max-height: 80vh;
            overflow-y: auto;
        }
        
        .channel-item {
            cursor: pointer;
            transition: all 0.2s;
            border-left: 3px solid transparent;
        }
        
        .channel-item:hover {
            background-color: rgba(255,255,255,0.1);
        }
        
        .channel-item.active {
            background-color: rgba(229,9,20,0.1);
            border-left: 3px solid var(--primary-color);
        }
        
        .channel-logo {
            width: 40px;
            height: 40px;
            object-fit: contain;
            border-radius: 4px;
        }
        
        .channel-logo-placeholder {
            width: 40px;
            height: 40px;
            display: flex;
            align-items: center;
            justify-content: center;
            background-color: rgba(255,255,255,0.1);
            border-radius: 4px;
        }
        
        .search-box {
            background-color: rgba(0,0,0,0.7);
            border: 1px solid #333;
            color: white;
        }
        
        .search-box:focus {
            background-color: rgba(0,0,0,0.9);
            color: white;
            border-color: #555;
            box-shadow: none;
        }
        
        .navbar {
            background-color: var(--dark-bg) !important;
            border-bottom: 1px solid #333;
        }
        
        .navbar-brand {
            color: var(--primary-color) !important;
            font-weight: bold;
        }
        
        /* Custom Alert Styles */
        #customAlert {
            min-width: 300px;
            max-width: 80%;
            opacity: 0;
            transition: opacity 0.3s ease;
        }
        
        #customAlert.show {
            opacity: 1;
        }
        
        /* Loading Spinner */
        #loadingSpinner {
            z-index: 9998;
        }
    </style>
</head>
<body>
    <!-- Simple Navbar -->
    <nav class="navbar navbar-expand-lg navbar-dark">
        <div class="container-fluid">
            <a class="navbar-brand" href="#">IPTV Player</a>
            <div class="d-flex">
                <div class="input-group me-3">
                    <input type="text" id="channelSearch" class="form-control search-box" placeholder="Search channels...">
                    <button class="btn btn-outline-secondary" type="button">
                        <i class="bi bi-search"></i>
                    </button>
                </div>
                <div class="dropdown">
                    <button class="btn btn-outline-danger dropdown-toggle" type="button" data-bs-toggle="dropdown">
                        <i class="bi bi-person-circle"></i>
                    </button>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <li><a class="dropdown-item" href="../dashboard.php">Dashboard</a></li>
                        <li><a class="dropdown-item" href="../subscriptions.php">Subscription</a></li>
                        <li><a class="dropdown-item" href="../wallet/">Wallet</a></li>
                        <li><hr class="dropdown-divider"></li>
                        <li><a class="dropdown-item" href="../logout.php">Logout</a></li>
                    </ul>
                </div>
            </div>
        </div>
    </nav>

    <!-- Custom Alert -->
    <div id="customAlert" class="alert alert-danger position-fixed top-0 start-50 translate-middle-x mt-3" style="display: none;">
        <div class="d-flex justify-content-between align-items-center">
            <span id="alertMessage"></span>
            <button type="button" class="btn-close" onclick="hideCustomAlert()"></button>
        </div>
    </div>

    <!-- Loading Spinner -->
    <div id="loadingSpinner" class="position-fixed top-50 start-50 translate-middle" style="display: none; z-index: 9998;">
        <div class="spinner-border text-danger" style="width: 3rem; height: 3rem;" role="status">
            <span class="visually-hidden">Loading...</span>
        </div>
    </div>

    <div class="container-fluid mt-3">
        <div class="row">
            <!-- Main Player Area -->
            <div class="col-lg-9">
                <div class="video-container mb-3">
                    <video id="iptvPlayer" controls autoplay playsinline></video>
                </div>
            </div>
            
            <!-- Channel List -->
            <div class="col-lg-3">
                <div class="card bg-dark">
                    <div class="card-header bg-dark">
                        <h5 class="mb-0">Channel List</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="list-group list-group-flush channel-list">
                            <?php foreach ($channels as $channel): ?>
                            <div class="list-group-item channel-item bg-dark text-white" data-url="<?= htmlspecialchars($channel['url']) ?>">
                                <div class="d-flex align-items-center">
                                    <?php if ($channel['logo']): ?>
                                    <img src="<?= htmlspecialchars($channel['logo']) ?>" alt="<?= htmlspecialchars($channel['name']) ?>" class="channel-logo me-3">
                                    <?php else: ?>
                                    <div class="channel-logo-placeholder me-3">
                                        <i class="bi bi-tv"></i>
                                    </div>
                                    <?php endif; ?>
                                    <span class="channel-name"><?= htmlspecialchars($channel['name']) ?></span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/hls.js@latest"></script>
    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const player = document.getElementById('iptvPlayer');
        const channelItems = document.querySelectorAll('.channel-item');
        const channelSearch = document.getElementById('channelSearch');
        let hls = new Hls();
        let retryCount = 0;
        const MAX_RETRIES = 3;
        
        // Play first channel by default
        if (channelItems.length > 0) {
            playChannel(channelItems[0]);
            channelItems[0].classList.add('active');
        }
        
        // Handle channel selection
        channelItems.forEach(item => {
            item.addEventListener('click', function() {
                showLoading();
                channelItems.forEach(i => i.classList.remove('active'));
                this.classList.add('active');
                playChannel(this);
            });
        });
        
        // Handle channel search
        channelSearch.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            channelItems.forEach(item => {
                const channelName = item.querySelector('.channel-name').textContent.toLowerCase();
                item.style.display = channelName.includes(searchTerm) ? 'flex' : 'none';
            });
        });
        
        // Custom alert functions
        function showCustomAlert(message, type = 'danger') {
            const alert = document.getElementById('customAlert');
            const alertMessage = document.getElementById('alertMessage');
            
            alert.className = `alert alert-${type} position-fixed top-0 start-50 translate-middle-x mt-3`;
            alertMessage.textContent = message;
            alert.style.display = 'block';
            
            // Auto-hide after 5 seconds
            setTimeout(hideCustomAlert, 5000);
        }
        
        function hideCustomAlert() {
            document.getElementById('customAlert').style.display = 'none';
        }
        
        // Loading functions
        function showLoading() {
            document.getElementById('loadingSpinner').style.display = 'block';
        }
        
        function hideLoading() {
            document.getElementById('loadingSpinner').style.display = 'none';
        }
        
        // Main playback function
        function playChannel(channelElement) {
            retryCount = 0;
            const streamUrl = channelElement.dataset.url;
            
            // Destroy previous HLS instance if exists
            if (hls) {
                hls.destroy();
            }
            
            // First try native HLS for Safari/iOS
            if (player.canPlayType('application/vnd.apple.mpegurl')) {
                player.src = streamUrl;
                player.load();
                
                const playPromise = player.play();
                
                if (playPromise !== undefined) {
                    playPromise.then(() => {
                        hideLoading();
                    }).catch(e => {
                        console.error('Native playback failed:', e);
                        tryHlsJsPlayback(streamUrl);
                    });
                }
            } 
            // Then try HLS.js for other browsers
            else if (Hls.isSupported()) {
                tryHlsJsPlayback(streamUrl);
            }
            // Final fallback
            else {
                player.src = streamUrl;
                player.load();
                player.play().then(() => {
                    hideLoading();
                }).catch(e => {
                    hideLoading();
                    showCustomAlert('Error: Your browser doesn\'t support HLS streaming. Try Chrome/Firefox or use VLC player.');
                });
            }
        }
        
        function tryHlsJsPlayback(streamUrl) {
            hls = new Hls({
                maxMaxBufferLength: 30,
                maxBufferSize: 60*1000*1000,
                maxBufferLength: 30,
                enableWorker: true,
                lowLatencyMode: true,
                backBufferLength: 30
            });
            
            hls.loadSource(streamUrl);
            hls.attachMedia(player);
            
            hls.on(Hls.Events.MANIFEST_PARSED, function() {
                player.play().then(() => {
                    hideLoading();
                }).catch(e => {
                    console.error('Auto-play failed:', e);
                    player.muted = true;
                    player.play().then(() => {
                        hideLoading();
                        player.muted = false;
                    }).catch(e => {
                        hideLoading();
                        showCustomAlert('Please click the play button to start playback');
                    });
                });
            });
            
            hls.on(Hls.Events.ERROR, function(event, data) {
                if (data.fatal) {
                    switch(data.type) {
                        case Hls.ErrorTypes.NETWORK_ERROR:
                            if (retryCount < MAX_RETRIES) {
                                retryCount++;
                                showCustomAlert(`Network error. Retrying (${retryCount}/${MAX_RETRIES})...`, 'warning');
                                setTimeout(() => hls.startLoad(), 2000 * retryCount);
                            } else {
                                hideLoading();
                                showCustomAlert('Failed to connect after multiple attempts. Please try again later.', 'danger');
                            }
                            break;
                        case Hls.ErrorTypes.MEDIA_ERROR:
                            showCustomAlert('Media error. Trying to recover...', 'warning');
                            hls.recoverMediaError();
                            break;
                        default:
                            hideLoading();
                            showCustomAlert('Unable to play this stream. Try another channel.', 'danger');
                            hls.destroy();
                    }
                }
            });
        }
        
        // Handle player errors
        player.addEventListener('error', function() {
            hideLoading();
            showCustomAlert('Error playing this channel. Please try another one.');
        });
    });
    </script>
</body>
</html>