<?php
// login.php

require_once __DIR__ . '/includes/config.php';
require_once __DIR__ . '/includes/functions.php';

// Initialize database connection
try {
    $pdo = getDBConnection();
} catch (PDOException $e) {
    error_log("Database connection failed: " . $e->getMessage());
    die("System temporarily unavailable. Please try again later.");
}

// Initialize secure session
if (session_status() === PHP_SESSION_NONE) {
    session_set_cookie_params([
        'lifetime' => 86400, // 1 day
        'path' => '/',
        'domain' => $_SERVER['HTTP_HOST'],
        'secure' => isset($_SERVER['HTTPS']),
        'httponly' => true,
        'samesite' => 'Lax'
    ]);
    session_start();
}

// Redirect if already logged in with valid session
if (isLoggedIn()) {
    if (checkSession()) {
        redirect('dashboard.php');
    } else {
        // Invalid session - destroy completely
        session_unset();
        session_destroy();
        setcookie(session_name(), '', time() - 3600, '/');
    }
}

// Initialize variables
$error = '';
$username = '';
$remember = false;

// Process login form
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // CSRF token validation
    if (empty($_POST['csrf_token']) || empty($_SESSION['csrf_token'])) {
        $error = "Security token missing. Please refresh the page.";
        unset($_SESSION['csrf_token']); // Clear invalid token
    } elseif (!hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $error = "Security token mismatch. Please refresh the page.";
        unset($_SESSION['csrf_token']); // Clear mismatched token
        error_log("CSRF token mismatch - IP: " . $_SERVER['REMOTE_ADDR'] . " - User: " . ($_POST['username'] ?? ''));
    } else {
        // Valid token - process login
        $username = sanitizeInput($_POST['username']);
        $password = $_POST['password'];
        $remember = isset($_POST['remember']);

        try {
            // Check login attempts
            if (checkLoginAttempts($username)) {
                $error = "Too many login attempts. Please try again in 15 minutes.";
            } else {
                $user = getUserByUsername($username);
                
                if ($user && password_verify($password, $user['password'])) {
                    // Successful login - regenerate session ID
                    session_regenerate_id(true);
                    
                    // Set fresh session data
                    $_SESSION = [
                        'user_id' => $user['id'],
                        'username' => $user['username'],
                        'last_activity' => time(),
                        'ip_address' => $_SERVER['REMOTE_ADDR'],
                        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
                        'csrf_token' => bin2hex(random_bytes(32))
                    ];

                    // Handle "Remember Me" functionality
                    if ($remember) {
                        $token = bin2hex(random_bytes(32));
                        $expiry = time() + (30 * 24 * 60 * 60); // 30 days
                        
                        setcookie(
                            'remember_token',
                            $user['id'] . ':' . $token,
                            [
                                'expires' => $expiry,
                                'path' => '/',
                                'domain' => $_SERVER['HTTP_HOST'],
                                'secure' => isset($_SERVER['HTTPS']),
                                'httponly' => true,
                                'samesite' => 'Lax'
                            ]
                        );
                        
                        // Store token in database
                        storeRememberToken($user['id'], $token, $expiry);
                    }
                    
                    // Update user last login and clear attempts
                    $pdo->beginTransaction();
                    try {
                        $stmt = $pdo->prepare("UPDATE users SET last_login = NOW() WHERE id = ?");
                        $stmt->execute([$user['id']]);
                        
                        $stmt = $pdo->prepare("DELETE FROM login_attempts WHERE username = ?");
                        $stmt->execute([$username]);
                        
                        $pdo->commit();
                    } catch (Exception $e) {
                        $pdo->rollBack();
                        error_log("Login update error: " . $e->getMessage());
                    }
                    
                    // Redirect to intended page
                    $redirect_url = $_SESSION['redirect_url'] ?? 'dashboard.php';
                    unset($_SESSION['redirect_url']);
                    redirect($redirect_url);
                } else {
                    $error = "Invalid username or password";
                    logFailedLoginAttempt($username, $_SERVER['REMOTE_ADDR']);
                    sleep(2); // Anti-brute force delay
                }
            }
        } catch (PDOException $e) {
            error_log("Login database error: " . $e->getMessage());
            $error = "System temporarily unavailable. Please try again later.";
        } catch (Exception $e) {
            error_log("Login error: " . $e->getMessage());
            $error = "An unexpected error occurred. Please try again.";
        }
    }
}

// Generate new CSRF token for form
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$pageTitle = "Login - " . SITE_NAME;
require_once __DIR__ . '/includes/header.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle, ENT_QUOTES, 'UTF-8') ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        :root {
            --primary-color: #4e73df;
            --secondary-color: #f8f9fc;
            --accent-color: #2e59d9;
            --text-color: #5a5c69;
            --light-text: #ffffff;
            --border-radius: 0.35rem;
        }
        
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: 'Nunito', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 
                        'Helvetica Neue', Arial, sans-serif;
            background-color: var(--secondary-color);
            color: var(--text-color);
            min-height: 100vh;
            display: flex;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }
        
        .login-container {
            width: 100%;
            max-width: 400px;
            margin: 0 auto;
        }
        
        .login-card {
            background: white;
            border-radius: var(--border-radius);
            box-shadow: 0 0.15rem 1.75rem 0 rgba(58, 59, 69, 0.15);
            padding: 2rem;
            text-align: center;
        }
        
        .login-logo {
            height: 60px;
            width: auto;
            margin-bottom: 1.5rem;
        }
        
        .login-title {
            font-size: 1.5rem;
            margin-bottom: 0.5rem;
            color: var(--text-color);
        }
        
        .login-subtitle {
            color: #6c757d;
            margin-bottom: 2rem;
            font-size: 0.9rem;
        }
        
        .form-group {
            margin-bottom: 1.5rem;
            text-align: left;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 0.5rem;
            font-weight: 600;
        }
        
        .form-control {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid #d1d3e2;
            border-radius: var(--border-radius);
            font-size: 1rem;
        }
        
        .form-control:focus {
            border-color: var(--primary-color);
            outline: none;
            box-shadow: 0 0 0 0.2rem rgba(78, 115, 223, 0.25);
        }
        
        .password-wrapper {
            position: relative;
        }
        
        .toggle-password {
            position: absolute;
            right: 10px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: #6c757d;
            cursor: pointer;
        }
        
        .form-options {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 1.5rem;
        }
        
        .remember-me {
            display: flex;
            align-items: center;
        }
        
        .remember-me input {
            margin-right: 0.5rem;
        }
        
        .forgot-password {
            color: var(--primary-color);
            text-decoration: none;
        }
        
        .btn {
            display: block;
            width: 100%;
            padding: 0.75rem;
            border-radius: var(--border-radius);
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .btn-primary {
            background-color: var(--primary-color);
            color: white;
            border: none;
        }
        
        .btn-primary:hover {
            background-color: var(--accent-color);
        }
        
        .login-footer {
            margin-top: 1.5rem;
            font-size: 0.9rem;
        }
        
        .login-footer a {
            color: var(--primary-color);
            text-decoration: none;
        }
        
        .alert {
            padding: 1rem;
            margin-bottom: 1.5rem;
            border-radius: var(--border-radius);
        }
        
        .alert-danger {
            background-color: #f8d7da;
            color: #721c24;
            border: 1px solid #f5c6cb;
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="login-card">
            <img src="<?= htmlspecialchars(SITE_LOGO, ENT_QUOTES, 'UTF-8') ?>" alt="<?= htmlspecialchars(SITE_NAME, ENT_QUOTES, 'UTF-8') ?>" class="login-logo">
            <h1 class="login-title">Welcome Back!</h1>
            <p class="login-subtitle">Please login to continue</p>
            
            <?php if ($error): ?>
                <div class="alert alert-danger">
                    <?= htmlspecialchars($error, ENT_QUOTES, 'UTF-8') ?>
                </div>
            <?php endif; ?>
            
            <form method="POST" action="login.php">
                <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($_SESSION['csrf_token'], ENT_QUOTES, 'UTF-8') ?>">
                
                <div class="form-group">
                    <label for="username">Username</label>
                    <input type="text" class="form-control" id="username" name="username" 
                           value="<?= htmlspecialchars($username, ENT_QUOTES, 'UTF-8') ?>" required autofocus>
                </div>
                
                <div class="form-group">
                    <label for="password">Password</label>
                    <div class="password-wrapper">
                        <input type="password" class="form-control" id="password" name="password" required>
                        <button type="button" class="toggle-password">
                            <i class="fas fa-eye"></i>
                        </button>
                    </div>
                </div>
                
                <div class="form-options">
                    <label class="remember-me">
                        <input type="checkbox" id="remember" name="remember" <?= $remember ? 'checked' : '' ?>>
                        Remember Me
                    </label>
                    <a href="forgot-password.php" class="forgot-password">Forgot Password?</a>
                </div>
                
                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-sign-in-alt"></i> Login
                </button>
            </form>
            
            <div class="login-footer">
                <p>Don't have an account? <a href="register.php">Create one</a></p>
            </div>
        </div>
    </div>

    <script>
        // Toggle password visibility
        document.querySelector('.toggle-password').addEventListener('click', function() {
            const password = document.getElementById('password');
            const icon = this.querySelector('i');
            const type = password.getAttribute('type') === 'password' ? 'text' : 'password';
            password.setAttribute('type', type);
            icon.classList.toggle('fa-eye');
            icon.classList.toggle('fa-eye-slash');
        });
    </script>
</body>
</html>

<?php
require_once __DIR__ . '/includes/footer.php';
?>