<?php
function processSubscriptionPurchase($user_id, $subscription) {
    global $pdo;
    
    $pdo->beginTransaction();
    
    try {
        // 1. Get current wallet balance
        $balance = getWalletBalance($user_id);
        
        // 2. Verify sufficient funds
        if ($balance < $subscription['price']) {
            return [
                'success' => false,
                'message' => 'Insufficient funds',
                'details' => 'Your current balance is not enough for this subscription',
                'required_amount' => $subscription['price'],
                'current_balance' => $balance
            ];
        }

        // 3. Deduct from wallet
        $stmt = $pdo->prepare("
            UPDATE wallet 
            SET balance = balance - ? 
            WHERE user_id = ? AND balance >= ?
        ");
        $stmt->execute([
            $subscription['price'],
            $user_id,
            $subscription['price']
        ]);
        
        if ($stmt->rowCount() === 0) {
            throw new Exception("Wallet deduction failed - possible race condition");
        }

        // 4. Record transaction
        $wallet_id = getOrCreateWallet($user_id);
        $stmt = $pdo->prepare("
            INSERT INTO wallet_transactions 
            (user_id, wallet_id, amount, transaction_type, reference, status)
            VALUES (?, ?, ?, ?, ?, 'completed')
        ");
        $stmt->execute([
            $user_id,
            $wallet_id,
            -$subscription['price'], // Negative for deduction
            'purchase',
            "Subscription: {$subscription['name']}"
        ]);
        $transaction_id = $pdo->lastInsertId();

        // 5. Expire old subscriptions
        $pdo->prepare("
            UPDATE user_subscriptions 
            SET status = 'expired' 
            WHERE user_id = ? AND status = 'active'
        ")->execute([$user_id]);

        // 6. Add new subscription
        $end_date = (new DateTime())
            ->add(new DateInterval("P{$subscription['duration_days']}D"))
            ->format('Y-m-d H:i:s');

        $stmt = $pdo->prepare("
            INSERT INTO user_subscriptions 
            (user_id, subscription_id, start_date, end_date, status)
            VALUES (?, ?, NOW(), ?, 'active')
        ");
        $stmt->execute([$user_id, $subscription['id'], $end_date]);

        $pdo->commit();

        return [
            'success' => true,
            'message' => 'Purchase successful',
            'subscription' => [
                'id' => $subscription['id'],
                'name' => $subscription['name'],
                'start_date' => date('Y-m-d H:i:s'),
                'end_date' => $end_date,
                'duration' => $subscription['duration_days']
            ],
            'balance' => $balance - $subscription['price'],
            'transaction_id' => $transaction_id
        ];

    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Purchase Processing Error [User: $user_id]: " . $e->getMessage());
        return [
            'success' => false,
            'message' => 'Failed to process purchase',
            'details' => $e->getMessage()
        ];
    }
}
?>