<?php
/**
 * ========================
 * SESSION FUNCTIONS
 * ========================
 */
function initializeSession() {
    if (session_status() === PHP_SESSION_NONE) {
        session_set_cookie_params([
            'lifetime' => SESSION_TIMEOUT,
            'path' => '/',
            'domain' => $_SERVER['HTTP_HOST'],
            'secure' => isset($_SERVER['HTTPS']),
            'httponly' => true,
            'samesite' => 'Lax'
        ]);
        session_start();
        
        if (!isset($_SESSION['csrf_token'])) {
            $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
            $_SESSION['csrf_token_time'] = time();
        }
        
        if (!isset($_SESSION['last_regeneration'])) {
            session_regenerate_id(true);
            $_SESSION['last_regeneration'] = time();
            $_SESSION['ip_address'] = $_SERVER['REMOTE_ADDR'];
            $_SESSION['user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';
        }
    }
}

/**
 * ========================
 * AUTHENTICATION FUNCTIONS
 * ========================
 */
function isLoggedIn() {
    if (!isset($_SESSION['user_id']) || !isset($_SESSION['last_activity'])) {
        return false;
    }
    
    if (time() - $_SESSION['last_activity'] > SESSION_TIMEOUT) {
        session_unset();
        session_destroy();
        return false;
    }
    
    $_SESSION['last_activity'] = time();
    return true;
}

function redirect($url, $permanent = false) {
    session_write_close();
    if ($permanent) {
        header("HTTP/1.1 301 Moved Permanently");
    }
    header("Location: $url");
    exit();
}

function sanitizeInput($data) {
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

/**
 * ========================
 * SECURITY FUNCTIONS
 * ========================
 */
function checkSession() {
    if (!isset($_SESSION['ip_address']) || $_SESSION['ip_address'] !== $_SERVER['REMOTE_ADDR']) {
        return false;
    }
    if (!isset($_SESSION['user_agent']) || $_SESSION['user_agent'] !== ($_SERVER['HTTP_USER_AGENT'] ?? '')) {
        return false;
    }
    if (!isset($_SESSION['last_activity']) || (time() - $_SESSION['last_activity'] > SESSION_TIMEOUT)) {
        return false;
    }
    $_SESSION['last_activity'] = time();
    return true;
}

function checkLoginAttempts($username) {
    global $pdo;
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as attempts 
        FROM login_attempts 
        WHERE username = ? AND attempt_time > DATE_SUB(NOW(), INTERVAL ? SECOND)
    ");
    $stmt->execute([$username, LOGIN_ATTEMPT_WINDOW]);
    $result = $stmt->fetch();
    return $result['attempts'] >= MAX_LOGIN_ATTEMPTS;
}

function logFailedLoginAttempt($username, $ip) {
    global $pdo;
    $stmt = $pdo->prepare("INSERT INTO login_attempts (username, ip_address) VALUES (?, ?)");
    $stmt->execute([$username, $ip]);
}

/**
 * ========================
 * REMEMBER ME FUNCTIONS
 * ========================
 */
function storeRememberToken($user_id, $token, $expiry) {
    global $pdo;
    $hashedToken = password_hash($token, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare("
        INSERT INTO remember_tokens 
        (user_id, token_hash, expires_at) 
        VALUES (?, ?, FROM_UNIXTIME(?))
    ");
    return $stmt->execute([$user_id, $hashedToken, $expiry]);
}

function validateRememberToken($token) {
    global $pdo;
    $stmt = $pdo->prepare("
        SELECT rt.*, u.* 
        FROM remember_tokens rt
        JOIN users u ON rt.user_id = u.id
        WHERE rt.expires_at > NOW()
    ");
    $stmt->execute();
    
    while ($row = $stmt->fetch()) {
        if (password_verify($token, $row['token_hash'])) {
            return $row;
        }
    }
    return false;
}

/**
 * ========================
 * USER MANAGEMENT
 * ========================
 */
function getUserById($id) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function getUserByUsername($username) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM users WHERE username = ?");
    $stmt->execute([$username]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * ========================
 * WALLET FUNCTIONS
 * ========================
 */
function getWalletBalance($user_id) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT COALESCE(balance, 0) as balance FROM wallet WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return (float)$result['balance'];
}

function addWalletTransaction($user_id, $amount, $type, $reference = null) {
    global $pdo;
    
    try {
        $wallet_id = getOrCreateWallet($user_id);
        
        $stmt = $pdo->prepare("
            INSERT INTO wallet_transactions 
            (user_id, wallet_id, amount, transaction_type, reference, status)
            VALUES (?, ?, ?, ?, ?, 'pending')
        ");
        $stmt->execute([$user_id, $wallet_id, $amount, $type, $reference]);
        
        return $pdo->lastInsertId();
    } catch (PDOException $e) {
        error_log("Wallet Transaction Error: " . $e->getMessage());
        return false;
    }
}

function getOrCreateWallet($user_id) {
    global $pdo;
    
    $stmt = $pdo->prepare("SELECT id FROM wallet WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $wallet = $stmt->fetch();
    
    if (!$wallet) {
        $pdo->prepare("INSERT INTO wallet (user_id, balance) VALUES (?, 0)")->execute([$user_id]);
        return $pdo->lastInsertId();
    }
    return $wallet['id'];
}

/**
 * ========================
 * SUBSCRIPTION FUNCTIONS
 * ========================
 */
function getAllSubscriptions() {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM subscriptions WHERE is_active = TRUE");
    $stmt->execute();
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function getUserSubscription($user_id) {
    global $pdo;
    $stmt = $pdo->prepare("
        SELECT us.*, s.name, s.description, s.price, s.duration_days 
        FROM user_subscriptions us
        JOIN subscriptions s ON us.subscription_id = s.id
        WHERE us.user_id = ? AND us.status = 'active' AND us.end_date > NOW()
        LIMIT 1
    ");
    $stmt->execute([$user_id]);
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

function purchaseSubscription($user_id, $subscription_id) {
    global $pdo;
    
    try {
        $pdo->beginTransaction();
        
        $stmt = $pdo->prepare("SELECT * FROM subscriptions WHERE id = ? AND is_active = TRUE");
        $stmt->execute([$subscription_id]);
        $subscription = $stmt->fetch();
        
        if (!$subscription) {
            throw new Exception("Subscription not available");
        }
        
        $stmt = $pdo->prepare("
            UPDATE wallet 
            SET balance = balance - ? 
            WHERE user_id = ? AND balance >= ?
        ");
        $stmt->execute([$subscription['price'], $user_id, $subscription['price']]);
        
        if ($stmt->rowCount() === 0) {
            throw new Exception("Insufficient funds");
        }
        
        $wallet_id = getOrCreateWallet($user_id);
        $stmt = $pdo->prepare("
            INSERT INTO wallet_transactions 
            (user_id, wallet_id, amount, transaction_type, reference, status)
            VALUES (?, ?, ?, ?, ?, 'completed')
        ");
        $stmt->execute([
            $user_id,
            $wallet_id,
            -$subscription['price'],
            'purchase',
            "Subscription: {$subscription['name']}"
        ]);
        
        $pdo->prepare("
            UPDATE user_subscriptions 
            SET status = 'expired' 
            WHERE user_id = ? AND status = 'active'
        ")->execute([$user_id]);
        
        $end_date = (new DateTime())
            ->add(new DateInterval("P{$subscription['duration_days']}D"))
            ->format('Y-m-d H:i:s');
            
        $pdo->prepare("
            INSERT INTO user_subscriptions 
            (user_id, subscription_id, start_date, end_date, status)
            VALUES (?, ?, NOW(), ?, 'active')
        ")->execute([$user_id, $subscription_id, $end_date]);
        
        $pdo->commit();
        return true;
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Purchase Error: " . $e->getMessage());
        return false;
    }
}

/**
 * ========================
 * M3U PLAYLIST FUNCTIONS
 * ========================
 */
function parseM3UPlaylist($url) {
    $channels = [];
    $content = @file_get_contents($url);
    
    if ($content === false) {
        return $channels;
    }
    
    $lines = explode("\n", $content);
    $currentChannel = null;
    
    foreach ($lines as $line) {
        $line = trim($line);
        
        if (strpos($line, '#EXTINF:') === 0) {
            $currentChannel = ['name' => '', 'logo' => '', 'url' => ''];
            preg_match('/#EXTINF:.*?,(.*)/', $line, $matches);
            if (isset($matches[1])) {
                $currentChannel['name'] = trim($matches[1]);
            }
            if (preg_match('/tvg-logo="([^"]+)"/', $line, $logoMatches)) {
                $currentChannel['logo'] = trim($logoMatches[1]);
            }
        } elseif ($line && $currentChannel && !str_starts_with($line, '#')) {
            $currentChannel['url'] = trim($line);
            $channels[] = $currentChannel;
            $currentChannel = null;
        }
    }
    
    return $channels;
}
/**
 * Store CSRF token in database
 */
function storeCsrfToken($token, $userId = null) {
    global $pdo;
    
    // Clear expired tokens first
    $pdo->exec("DELETE FROM csrf_tokens WHERE expires_at < NOW()");
    
    $hashedToken = hash('sha256', $token);
    $stmt = $pdo->prepare("
        INSERT INTO csrf_tokens 
        (user_id, token_hash, expires_at)
        VALUES (?, ?, DATE_ADD(NOW(), INTERVAL 1 HOUR))
    ");
    return $stmt->execute([$userId, $hashedToken]);
}

/**
 * Validate and clear CSRF token
 */
function validateCsrfToken($token, $userId = null) {
    global $pdo;
    
    $hashedToken = hash('sha256', $token);
    
    // Find and delete the token in one operation
    $stmt = $pdo->prepare("
        DELETE FROM csrf_tokens 
        WHERE token_hash = ? 
        AND (user_id = ? OR (? IS NULL AND user_id IS NULL))
        AND (expires_at > NOW() OR expires_at IS NULL)
        LIMIT 1
    ");
    $stmt->execute([$hashedToken, $userId, $userId]);
    
    return $stmt->rowCount() === 1;
}

/**
 * Clear all CSRF tokens for a user
 */
function clearUserCsrfTokens($userId) {
    global $pdo;
    $stmt = $pdo->prepare("DELETE FROM csrf_tokens WHERE user_id = ?");
    return $stmt->execute([$userId]);
}
// Initialize session
initializeSession();