<?php
// dashboard.php

require_once 'includes/config.php';
require_once 'includes/functions.php';

// Enhanced session configuration with cookie settings
if (session_status() === PHP_SESSION_NONE) {
    session_set_cookie_params([
        'lifetime' => 86400, // 1 day
        'path' => '/',
        'domain' => $_SERVER['HTTP_HOST'],
        'secure' => isset($_SERVER['HTTPS']),
        'httponly' => true,
        'samesite' => 'Lax' // Changed from Strict to Lax for better compatibility
    ]);
    
    session_start([
        'use_strict_mode' => true,
        'use_cookies' => true,
        'cookie_lifetime' => 86400,
        'cookie_secure' => isset($_SERVER['HTTPS']),
        'cookie_httponly' => true,
        'cookie_samesite' => 'Lax'
    ]);
}

// Generate CSRF token if doesn't exist
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// [Rest of your existing security checks remain the same...]

// Redirect if not logged in or session is invalid
if (!isLoggedIn() || !checkSession()) {
    // Store current URL for redirect after login
    $_SESSION['redirect_url'] = $_SERVER['REQUEST_URI'];
    redirect('login.php');
    exit;
}

// Check for session hijacking
if ($_SESSION['ip_address'] !== $_SERVER['REMOTE_ADDR'] || 
    $_SESSION['user_agent'] !== $_SERVER['HTTP_USER_AGENT']) {
    session_destroy();
    redirect('login.php');
    exit;
}

// Update last activity time
$_SESSION['last_activity'] = time();

// Check for idle timeout (30 minutes)
if (isset($_SESSION['last_activity']) && 
    (time() - $_SESSION['last_activity'] > 1800)) {
    session_destroy();
    redirect('login.php?timeout=1');
    exit;
}

// Get user data
try {
    $user = getUserById($_SESSION['user_id']);
    if (!$user) {
        throw new Exception("User not found");
    }
    
    $wallet_balance = getWalletBalance($_SESSION['user_id']);
    $subscription = getUserSubscription($_SESSION['user_id']);
    
    // Regenerate session ID periodically
    if (!isset($_SESSION['regenerated'])) {
        session_regenerate_id(true);
        $_SESSION['regenerated'] = time();
    } elseif (time() - $_SESSION['regenerated'] > 1800) {
        session_regenerate_id(true);
        $_SESSION['regenerated'] = time();
        // Regenerate CSRF token when session ID changes
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    
} catch (Exception $e) {
    error_log("Dashboard error: " . $e->getMessage());
    session_destroy();
    redirect('login.php?error=1');
    exit;
}

$pageTitle = "Dashboard - " . SITE_NAME;
require_once 'includes/header.php';
?>

<style>
    /* Dashboard Specific Styles */
    .dashboard-container {
        display: flex;
        min-height: calc(100vh - 70px); /* Adjust for navbar height */
        margin-top: 70px; /* Space for fixed navbar */
    }
    
    .sidebar {
        width: 250px;
        background-color: #2c3e50;
        color: white;
        padding: 1.5rem 0;
        position: fixed;
        height: calc(100vh - 70px);
    }
    
    .user-profile {
        text-align: center;
        padding: 0 1rem 1.5rem;
        border-bottom: 1px solid rgba(255,255,255,0.1);
    }
    
    .avatar {
        font-size: 4rem;
        color: #bdc3c7;
        margin-bottom: 0.5rem;
    }
    
    .sidebar-nav {
        margin-top: 1rem;
    }
    
    .sidebar-nav a {
        display: block;
        color: #ecf0f1;
        padding: 0.75rem 1.5rem;
        text-decoration: none;
        transition: all 0.3s;
    }
    
    .sidebar-nav a:hover, 
    .sidebar-nav a.active {
        background-color: #34495e;
        color: white;
    }
    
    .sidebar-nav i {
        width: 20px;
        margin-right: 0.5rem;
        text-align: center;
    }
    
    .main-content {
        flex: 1;
        margin-left: 250px; /* Same as sidebar width */
        padding: 2rem;
    }
    
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 1.5rem;
        margin-bottom: 2rem;
    }
    
    .stat-card {
        background: white;
        border-radius: 8px;
        padding: 1.5rem;
        display: flex;
        align-items: center;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
    }
    
    .stat-icon {
        font-size: 2rem;
        color: var(--primary-color);
        margin-right: 1.5rem;
    }
    
    .stat-info h4 {
        color: #7f8c8d;
        font-size: 0.9rem;
        margin-bottom: 0.5rem;
    }
    
    .stat-info h2 {
        font-size: 1.8rem;
        color: var(--text-color);
    }
    
    .card {
        background: white;
        border-radius: 8px;
        box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        margin-bottom: 2rem;
    }
    
    .card-header {
        padding: 1rem 1.5rem;
        border-bottom: 1px solid #eee;
    }
    
    .card-header h3 {
        font-size: 1.2rem;
        color: var(--text-color);
    }
    
    .card-body {
        padding: 1.5rem;
    }
    
    .alert {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding: 1rem;
        border-radius: 6px;
        margin-bottom: 1.5rem;
    }
    
    .alert.success {
        background-color: #e8f5e9;
        color: #2e7d32;
    }
    
    .alert.warning {
        background-color: #fff8e1;
        color: #ff8f00;
    }
    
    .alert-content h4 {
        margin-bottom: 0.3rem;
    }
    
    .status-badge {
        padding: 0.3rem 0.8rem;
        border-radius: 20px;
        font-size: 0.8rem;
        font-weight: 600;
    }
    
    .status-badge.active {
        background-color: #2e7d32;
        color: white;
    }
    
    .status-badge.inactive {
        background-color: #ff8f00;
        color: white;
    }
    
    .btn {
        display: inline-flex;
        align-items: center;
        padding: 0.75rem 1.5rem;
        border-radius: 6px;
        font-weight: 500;
        text-decoration: none;
        transition: all 0.3s;
    }
    
    .btn i {
        margin-right: 0.5rem;
    }
    
    .btn.primary {
        background-color: var(--primary-color);
        color: white;
    }
    
    .btn.primary:hover {
        background-color: var(--accent-color);
    }
    
    .quick-actions {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 1.5rem;
    }
    
    .action-btn {
        background-color: #f5f5f5;
        color: var(--text-color);
        margin-bottom: 0.75rem;
    }
    
    .action-btn:hover {
        background-color: #e0e0e0;
    }
    
    .activity-list {
        display: flex;
        flex-direction: column;
        gap: 1rem;
    }
    
    .activity-item {
        display: flex;
        align-items: center;
        padding: 0.75rem;
        border-radius: 6px;
        background-color: #f9f9f9;
    }
    
    .activity-icon {
        width: 36px;
        height: 36px;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin-right: 1rem;
    }
    
    .activity-icon.success {
        background-color: #e8f5e9;
        color: #2e7d32;
    }
    
    .activity-icon.danger {
        background-color: #ffebee;
        color: #c62828;
    }
    
    .activity-content {
        flex: 1;
    }
    
    .activity-header {
        display: flex;
        justify-content: space-between;
        margin-bottom: 0.2rem;
    }
    
    .amount.success {
        color: #2e7d32;
    }
    
    .amount.danger {
        color: #c62828;
    }
    
    @media (max-width: 768px) {
        .dashboard-container {
            flex-direction: column;
        }
        
        .sidebar {
            width: 100%;
            position: static;
            height: auto;
        }
        
        .main-content {
            margin-left: 0;
            padding: 1rem;
        }
        
        .quick-actions {
            grid-template-columns: 1fr;
        }
    }
</style>

<div class="dashboard-container">
    <div class="sidebar">
        <div class="user-profile">
            <div class="avatar">
                <i class="fas fa-user-circle"></i>
            </div>
            <h3><?= htmlspecialchars($user['username'], ENT_QUOTES, 'UTF-8') ?></h3>
            <p><?= htmlspecialchars($user['email'], ENT_QUOTES, 'UTF-8') ?></p>
        </div>
        
        <nav class="sidebar-nav">
            <a href="dashboard.php" class="active">
                <i class="fas fa-tachometer-alt"></i> Dashboard
            </a>
            <a href="player/">
                <i class="fas fa-play-circle"></i> Watch TV
            </a>
            <a href="subscriptions.php">
                <i class="fas fa-tv"></i> Subscriptions
            </a>
            <a href="wallet/">
                <i class="fas fa-wallet"></i> Wallet
            </a>
            <a href="logout.php?csrf_token=<?= htmlspecialchars($_SESSION['csrf_token'], ENT_QUOTES, 'UTF-8') ?>">
                <i class="fas fa-sign-out-alt"></i> Logout
            </a>
        </nav>
    </div>
    
    <div class="main-content">
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-wallet"></i>
                </div>
                <div class="stat-info">
                    <h4>Wallet Balance</h4>
                    <h2>Nrs. <?= number_format($wallet_balance, 2) ?></h2>
                </div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon">
                    <i class="fas fa-tv"></i>
                </div>
                <div class="stat-info">
                    <h4>Subscription</h4>
                    <h2><?= $subscription ? htmlspecialchars($subscription['name'], ENT_QUOTES, 'UTF-8') : 'None' ?></h2>
                </div>
            </div>
        </div>
        
        <div class="card">
            <div class="card-header">
                <h3>Subscription Status</h3>
            </div>
            <div class="card-body">
                <?php if ($subscription): ?>
                    <div class="alert success">
                        <div class="alert-content">
                            <h4><?= htmlspecialchars($subscription['name'], ENT_QUOTES, 'UTF-8') ?></h4>
                            <p>Expires on: <?= date('M d, Y', strtotime($subscription['end_date'])) ?></p>
                        </div>
                        <span class="status-badge active">Active</span>
                    </div>
                    <a href="player/" class="btn primary">
                        <i class="fas fa-play-circle"></i> Watch TV Now
                    </a>
                <?php else: ?>
                    <div class="alert warning">
                        <div class="alert-content">
                            <h4>No Active Subscription</h4>
                            <p>You don't have an active subscription. Purchase one to start watching.</p>
                        </div>
                        <span class="status-badge inactive">Inactive</span>
                    </div>
                    <a href="subscriptions.php" class="btn primary">
                        <i class="fas fa-shopping-cart"></i> Buy Subscription
                    </a>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="quick-actions">
            <div class="card">
                <div class="card-header">
                    <h3>Quick Actions</h3>
                </div>
                <div class="card-body">
                    <a href="wallet/" class="btn action-btn">
                        <i class="fas fa-plus-circle"></i> Add Funds
                    </a>
                    <a href="subscriptions.php" class="btn action-btn">
                        <i class="fas fa-shopping-cart"></i> Buy Subscription
                    </a>
                    <a href="player/" class="btn action-btn">
                        <i class="fas fa-play-circle"></i> Watch TV
                    </a>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h3>Recent Activity</h3>
                </div>
                <div class="card-body">
                    <?php
                    try {
                        $stmt = $pdo->prepare("
                            SELECT wt.* 
                            FROM wallet_transactions wt
                            JOIN wallet w ON wt.wallet_id = w.id
                            WHERE w.user_id = ?
                            ORDER BY wt.created_at DESC
                            LIMIT 3
                        ");
                        $stmt->execute([$_SESSION['user_id']]);
                        $transactions = $stmt->fetchAll(PDO::FETCH_ASSOC);
                        
                        if (empty($transactions)) {
                            echo '<p class="text-muted">No recent activity</p>';
                        } else {
                            echo '<div class="activity-list">';
                            foreach ($transactions as $transaction) {
                                $type_class = $transaction['transaction_type'] === 'deposit' ? 'success' : 'danger';
                                $icon = $transaction['transaction_type'] === 'deposit' ? 'plus-circle' : 'minus-circle';
                                echo '
                                <div class="activity-item">
                                    <div class="activity-icon ' . htmlspecialchars($type_class, ENT_QUOTES, 'UTF-8') . '">
                                        <i class="fas fa-' . htmlspecialchars($icon, ENT_QUOTES, 'UTF-8') . '"></i>
                                    </div>
                                    <div class="activity-content">
                                        <div class="activity-header">
                                            <span>' . htmlspecialchars(ucfirst($transaction['transaction_type']), ENT_QUOTES, 'UTF-8') . '</span>
                                            <span class="amount ' . htmlspecialchars($type_class, ENT_QUOTES, 'UTF-8') . '">Nrs. ' . htmlspecialchars(number_format($transaction['amount'], 2), ENT_QUOTES, 'UTF-8') . '</span>
                                        </div>
                                        <small>' . htmlspecialchars(date('M d, H:i', strtotime($transaction['created_at'])), ENT_QUOTES, 'UTF-8') . '</small>
                                    </div>
                                </div>
                                ';
                            }
                            echo '</div>';
                        }
                    } catch (PDOException $e) {
                        error_log("Database error: " . $e->getMessage());
                        echo '<p class="text-muted">Unable to load recent activity</p>';
                    }
                    ?>
                </div>
            </div>
        </div>
    </div>
</div>

<?php 
require_once 'includes/footer.php';
?>