<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/functions.php';

header('Content-Type: application/json');

try {
    // Check authentication
    if (!isLoggedIn()) {
        throw new Exception('Authentication required', 401);
    }

    $user_id = $_SESSION['user_id'];

    // Handle different HTTP methods
    switch ($_SERVER['REQUEST_METHOD']) {
        case 'GET':
            // Get all available subscriptions
            $stmt = $pdo->prepare("
                SELECT s.id, s.name, s.description, s.price, s.duration_days,
                       (us.user_id IS NOT NULL AND us.status = 'active' AND us.end_date > NOW()) as is_active
                FROM subscriptions s
                LEFT JOIN user_subscriptions us ON s.id = us.subscription_id AND us.user_id = ?
                WHERE s.is_active = TRUE
            ");
            $stmt->execute([$user_id]);
            $subscriptions = $stmt->fetchAll(PDO::FETCH_ASSOC);

            // Get current subscription if any
            $current_sub = getUserSubscription($user_id);

            echo json_encode([
                'success' => true,
                'data' => [
                    'subscriptions' => $subscriptions,
                    'current_subscription' => $current_sub,
                    'wallet_balance' => getWalletBalance($user_id)
                ]
            ]);
            break;

        case 'POST':
            // Handle subscription purchase
            $data = json_decode(file_get_contents('php://input'), true);
            $subscription_id = filter_var($data['subscription_id'] ?? 0, FILTER_VALIDATE_INT);

            // Validate subscription ID
            if ($subscription_id === false || $subscription_id <= 0) {
                throw new Exception('Invalid subscription ID', 400);
            }

            // Verify subscription exists and is active
            $stmt = $pdo->prepare("SELECT * FROM subscriptions WHERE id = ? AND is_active = TRUE LIMIT 1");
            $stmt->execute([$subscription_id]);
            $subscription = $stmt->fetch();

            if (!$subscription) {
                throw new Exception('Subscription not available', 404);
            }

            // Process purchase
            $pdo->beginTransaction();

            // 1. Check balance
            $balance = getWalletBalance($user_id);
            if ($balance < $subscription['price']) {
                throw new Exception(json_encode([
                    'message' => 'Insufficient funds',
                    'required' => $subscription['price'],
                    'current' => $balance
                ]), 402);
            }

            // 2. Deduct from wallet
            $stmt = $pdo->prepare("
                UPDATE wallet 
                SET balance = balance - ? 
                WHERE user_id = ? AND balance >= ?
            ");
            $stmt->execute([$subscription['price'], $user_id, $subscription['price']]);
            
            if ($stmt->rowCount() === 0) {
                throw new Exception('Wallet update failed', 400);
            }

            // 3. Record transaction
            $wallet_id = getOrCreateWallet($user_id);
            $stmt = $pdo->prepare("
                INSERT INTO wallet_transactions 
                (user_id, wallet_id, amount, transaction_type, reference, status)
                VALUES (?, ?, ?, ?, ?, 'completed')
            ");
            $stmt->execute([
                $user_id,
                $wallet_id,
                $subscription['price'],
                'purchase',
                "Subscription: {$subscription['name']}"
            ]);

            // 4. Expire old subscriptions
            $pdo->prepare("
                UPDATE user_subscriptions 
                SET status = 'expired' 
                WHERE user_id = ? AND status = 'active'
            ")->execute([$user_id]);

            // 5. Add new subscription
            $end_date = date('Y-m-d H:i:s', strtotime("+{$subscription['duration_days']} days"));
            $stmt = $pdo->prepare("
                INSERT INTO user_subscriptions 
                (user_id, subscription_id, end_date, status)
                VALUES (?, ?, ?, 'active')
            ");
            $stmt->execute([$user_id, $subscription['id'], $end_date]);

            $pdo->commit();

            echo json_encode([
                'success' => true,
                'message' => 'Subscription purchased successfully',
                'data' => [
                    'subscription' => [
                        'id' => $subscription['id'],
                        'name' => $subscription['name'],
                        'end_date' => $end_date
                    ],
                    'new_balance' => $balance - $subscription['price']
                ]
            ]);
            break;

        default:
            throw new Exception('Method not allowed', 405);
    }

} catch (Exception $e) {
    $code = $e->getCode() ?: 500;
    http_response_code($code);

    $message = $e->getMessage();
    $decoded = json_decode($message, true);
    
    echo json_encode([
        'success' => false,
        'message' => $decoded ? $decoded['message'] ?? $message : $message,
        'details' => $decoded ?? null
    ]);
}

function getOrCreateWallet($user_id) {
    global $pdo;
    
    $stmt = $pdo->prepare("SELECT id FROM wallet WHERE user_id = ? LIMIT 1");
    $stmt->execute([$user_id]);
    $wallet = $stmt->fetch();
    
    if ($wallet) {
        return $wallet['id'];
    }
    
    $pdo->prepare("INSERT INTO wallet (user_id) VALUES (?)")->execute([$user_id]);
    return $pdo->lastInsertId();
}
?>