<?php
declare(strict_types=1);

header('Content-Type: application/json');

// Define base path
$basePath = '/home/chimchim/public_html/IPTV/root';
require_once $basePath . '/includes/config.php';
require_once $basePath . '/includes/functions.php';

// Verify request method
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Only POST requests are allowed']);
    exit();
}

// Check authentication
if (!isLoggedIn()) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Authentication required']);
    exit();
}

// Get and validate input
$input = json_decode(file_get_contents('php://input'), true);
if (empty($input['subscription_id'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Subscription ID is required']);
    exit();
}

$subscription_id = (int)$input['subscription_id'];
$transactionActive = false;

try {
    // Start transaction
    $pdo->beginTransaction();
    $transactionActive = true;

    // 1. Verify subscription exists in subscriptions table and get details
    $stmt = $pdo->prepare("
        SELECT id, name, price, duration_days 
        FROM subscriptions 
        WHERE id = ?
        LIMIT 1
    ");
    $stmt->execute([$subscription_id]);
    $subscription = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$subscription) {
        throw new Exception(json_encode([
            'success' => false,
            'message' => 'Subscription not found',
            'subscription_id' => $subscription_id
        ]));
    }

    // 2. Ensure record exists in subscription_plans (for foreign key)
    $stmt = $pdo->prepare("
        INSERT INTO subscription_plans (id, name, price, duration_days)
        SELECT id, name, price, duration_days 
        FROM subscriptions 
        WHERE id = ?
        ON DUPLICATE KEY UPDATE
            name = VALUES(name),
            price = VALUES(price),
            duration_days = VALUES(duration_days)
    ");
    $stmt->execute([$subscription_id]);

    // 3. Check wallet balance
    $balance = getWalletBalance($_SESSION['user_id']);
    if ($balance === false || $balance < $subscription['price']) {
        throw new Exception(json_encode([
            'success' => false,
            'message' => 'Insufficient funds',
            'required' => (float)$subscription['price'],
            'current_balance' => (float)$balance
        ]));
    }

    // 4. Deduct from wallet
    $stmt = $pdo->prepare("
        UPDATE wallet 
        SET balance = balance - ? 
        WHERE user_id = ? AND balance >= ?
    ");
    $stmt->execute([$subscription['price'], $_SESSION['user_id'], $subscription['price']]);

    if ($stmt->rowCount() === 0) {
        throw new Exception("Failed to deduct from wallet");
    }

    // 5. Record transaction
    $wallet_id = getOrCreateWallet($_SESSION['user_id']);
    $stmt = $pdo->prepare("
        INSERT INTO wallet_transactions 
        (user_id, wallet_id, amount, transaction_type, reference, status)
        VALUES (?, ?, ?, 'purchase', ?, 'completed')
    ");
    $stmt->execute([
        $_SESSION['user_id'],
        $wallet_id,
        -$subscription['price'],
        "Subscription: {$subscription['name']}"
    ]);

    // 6. Expire old subscriptions
    $pdo->prepare("
        UPDATE user_subscriptions 
        SET status = 'expired' 
        WHERE user_id = ? AND status = 'active'
    ")->execute([$_SESSION['user_id']]);

    // 7. Add new subscription
    $end_date = (new DateTime())->add(new DateInterval("P{$subscription['duration_days']}D"))->format('Y-m-d H:i:s');
    
    $stmt = $pdo->prepare("
        INSERT INTO user_subscriptions 
        (user_id, subscription_id, start_date, end_date, status)
        VALUES (?, ?, NOW(), ?, 'active')
    ");
    $stmt->execute([$_SESSION['user_id'], $subscription['id'], $end_date]);

    // Commit transaction
    $pdo->commit();
    $transactionActive = false;

    // Success response with all required data
    $new_balance = $balance - $subscription['price'];
    echo json_encode([
        'success' => true,
        'message' => 'Subscription purchased successfully',
        'data' => [
            'subscription' => [
                'id' => (int)$subscription['id'],
                'name' => $subscription['name'],
                'price' => (float)$subscription['price'],
                'duration_days' => (int)$subscription['duration_days'],
                'start_date' => date('Y-m-d H:i:s'),
                'end_date' => $end_date
            ],
            'balance' => (float)$new_balance,
            'new_balance' => (float)$new_balance
        ]
    ]);

} catch (PDOException $e) {
    if ($transactionActive) {
        $pdo->rollBack();
    }
    
    if ($e->getCode() == '23000') {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Database constraint violation',
            'error' => $e->getMessage(),
            'solution' => 'The subscription might not exist in subscription_plans table'
        ]);
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Database error',
            'error' => $e->getMessage(),
            'code' => $e->getCode()
        ]);
    }
} catch (Exception $e) {
    if ($transactionActive) {
        $pdo->rollBack();
    }
    $errorData = json_decode($e->getMessage(), true);
    if (json_last_error() === JSON_ERROR_NONE) {
        echo json_encode($errorData);
    } else {
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Operation failed',
            'error' => $e->getMessage()
        ]);
    }
}