<?php
// api/channels.php

require_once '../includes/config.php';

// Set JSON content type
header('Content-Type: application/json');
header('Cache-Control: private, max-age=3600'); // Cache for 1 hour

// Verify authentication and subscription
if (!isLoggedIn()) {
    http_response_code(401);
    die(json_encode(['error' => 'Unauthorized', 'code' => 401]));
}

$subscription = getUserSubscription($_SESSION['user_id']);
if (!$subscription) {
    http_response_code(403);
    die(json_encode(['error' => 'Subscription required', 'code' => 403]));
}

// Only allow GET requests
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    die(json_encode([
        'error' => 'Method not allowed',
        'code' => 405,
        'allowed_methods' => ['GET']
    ]));
}

try {
    // Cache handling
    $cacheKey = 'channels_' . md5(IPTV_PLAYLIST_URL);
    $cachedChannels = apcu_fetch($cacheKey);
    
    if ($cachedChannels !== false) {
        echo json_encode([
            'success' => true,
            'cached' => true,
            'channels' => $cachedChannels
        ]);
        exit();
    }

    // Parse channels with error handling
    $channels = parseM3UPlaylist(IPTV_PLAYLIST_URL);
    
    if (empty($channels)) {
        throw new Exception('No channels found in playlist');
    }

    // Cache the results for 1 hour
    apcu_store($cacheKey, $channels, 3600);
    
    echo json_encode([
        'success' => true,
        'cached' => false,
        'channels' => $channels,
        'count' => count($channels),
        'generated_at' => date('c')
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Failed to load channels',
        'message' => $e->getMessage(),
        'code' => 500
    ]);
}