<?php
// admin_panel_installer.php
// Run this script ONCE to create your complete admin panel
// Delete this file after installation for security

// Configuration
$adminFolder = 'admin';
$includesFolder = 'includes';
$assetsFolder = 'assets/css';
$siteName = 'My Admin Panel'; // Change to your site name

// Create folders if they don't exist
$folders = [$adminFolder, "$adminFolder/includes", $includesFolder, $assetsFolder];
foreach ($folders as $folder) {
    if (!file_exists($folder)) {
        mkdir($folder, 0755, true);
        echo "Created folder: $folder<br>";
    }
}

// 1. Create config.php
file_put_contents("$includesFolder/config.php", <<<'EOT'
<?php
// Database Configuration
define('DB_HOST', 'localhost');
define('DB_NAME', 'your_database');
define('DB_USER', 'your_username');
define('DB_PASS', 'your_password');

// Site Configuration
define('SITE_NAME', 'My Admin Panel');
define('SITE_URL', 'http://yourdomain.com');
define('SITE_LOGO', '/assets/img/logo.png');

// Error Reporting (disable in production)
error_reporting(E_ALL);
ini_set('display_errors', 1);

// Session Configuration
session_set_cookie_params([
    'lifetime' => 86400,
    'path' => '/',
    'domain' => $_SERVER['HTTP_HOST'],
    'secure' => isset($_SERVER['HTTPS']),
    'httponly' => true,
    'samesite' => 'Strict'
]);

// Create database connection
try {
    $pdo = new PDO("mysql:host=".DB_HOST.";dbname=".DB_NAME, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("Database connection failed: " . $e->getMessage());
}

// Basic functions
function sanitizeInput($data) {
    return htmlspecialchars(strip_tags(trim($data)));
}

function redirect($url) {
    header("Location: $url");
    exit();
}
EOT
);

// 2. Create functions.php
file_put_contents("$includesFolder/functions.php", <<<'EOT'
<?php
require_once __DIR__ . '/config.php';

function isLoggedIn() {
    return isset($_SESSION['user_id']);
}

function getUserById($id) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$id]);
    return $stmt->fetch();
}

function getWalletBalance($user_id) {
    global $pdo;
    $stmt = $pdo->prepare("SELECT balance FROM wallet WHERE user_id = ?");
    $stmt->execute([$user_id]);
    return $stmt->fetchColumn() ?? 0.00;
}

function addWalletTransaction($user_id, $amount, $type, $reference, $status = 'pending') {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT id FROM wallet WHERE user_id = ?");
        $stmt->execute([$user_id]);
        $wallet_id = $stmt->fetchColumn();
        
        if (!$wallet_id) return false;
        
        $stmt = $pdo->prepare("
            INSERT INTO wallet_transactions 
            (wallet_id, amount, transaction_type, reference, status) 
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([$wallet_id, $amount, $type, $reference, $status]);
        
        return $pdo->lastInsertId();
    } catch (PDOException $e) {
        error_log("Transaction error: " . $e->getMessage());
        return false;
    }
}
EOT
);

// 3. Create admin_functions.php
file_put_contents("$includesFolder/admin_functions.php", <<<'EOT'
<?php
function isAdminLoggedIn() {
    if (session_status() === PHP_SESSION_NONE) session_start();
    return isset($_SESSION['admin_logged_in']) && $_SESSION['admin_logged_in'] === true;
}

function adminLogin($username, $password) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT * FROM admin_users WHERE username = ? AND is_active = TRUE");
        $stmt->execute([$username]);
        $admin = $stmt->fetch();
        
        if ($admin && password_verify($password, $admin['password_hash'])) {
            session_regenerate_id(true);
            $_SESSION['admin_id'] = $admin['id'];
            $_SESSION['admin_username'] = $admin['username'];
            $_SESSION['admin_logged_in'] = true;
            
            $pdo->prepare("UPDATE admin_users SET last_login = NOW() WHERE id = ?")
               ->execute([$admin['id']]);
            return true;
        }
    } catch (PDOException $e) {
        error_log("Admin login error: " . $e->getMessage());
    }
    return false;
}

function adminLogout() {
    $_SESSION = [];
    if (ini_get("session.use_cookies")) {
        $params = session_get_cookie_params();
        setcookie(session_name(), '', time() - 42000,
            $params["path"], $params["domain"],
            $params["secure"], $params["httponly"]
        );
    }
    session_destroy();
}

function getPendingDeposits() {
    global $pdo;
    $stmt = $pdo->query("
        SELECT wt.*, u.username 
        FROM wallet_transactions wt
        JOIN wallet w ON wt.wallet_id = w.id
        JOIN users u ON w.user_id = u.id
        WHERE wt.status = 'pending' AND wt.transaction_type = 'deposit'
        ORDER BY wt.created_at DESC
    ");
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

function processWalletTransaction($transaction_id, $action) {
    global $pdo;
    
    try {
        $pdo->beginTransaction();
        $status = ($action === 'approve') ? 'completed' : 'rejected';
        $stmt = $pdo->prepare("UPDATE wallet_transactions SET status = ? WHERE id = ?");
        $stmt->execute([$status, $transaction_id]);
        
        if ($action === 'approve') {
            $stmt = $pdo->prepare("
                UPDATE wallet w
                JOIN wallet_transactions wt ON w.id = wt.wallet_id
                SET w.balance = w.balance + wt.amount
                WHERE wt.id = ? AND wt.status = 'completed'
            ");
            $stmt->execute([$transaction_id]);
        }
        
        $pdo->commit();
        return true;
    } catch (PDOException $e) {
        $pdo->rollBack();
        error_log("Transaction error: " . $e->getMessage());
        return false;
    }
}
EOT
);

// 4. Create admin/login.php
file_put_contents("$adminFolder/login.php", <<<'EOT'
<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/admin_functions.php';

if (isAdminLoggedIn()) redirect('index.php');

$error = '';
$username = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username = sanitizeInput($_POST['username']);
    $password = $_POST['password'];
    
    if (adminLogin($username, $password)) {
        redirect('index.php');
    } else {
        $error = 'Invalid credentials';
        sleep(1);
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Login - <?= SITE_NAME ?></title>
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body>
    <div class="login-container">
        <div class="login-box">
            <h1><i class="fas fa-lock"></i> Admin Login</h1>
            <?php if ($error): ?><div class="alert error"><?= $error ?></div><?php endif; ?>
            <form method="POST">
                <div class="form-group">
                    <label>Username</label>
                    <input type="text" name="username" value="<?= htmlspecialchars($username) ?>" required>
                </div>
                <div class="form-group">
                    <label>Password</label>
                    <input type="password" name="password" required>
                </div>
                <button type="submit" class="btn">Login</button>
            </form>
        </div>
    </div>
</body>
</html>
EOT
);

// 5. Create admin/index.php
file_put_contents("$adminFolder/index.php", <<<'EOT'
<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/admin_functions.php';

if (!isAdminLoggedIn()) redirect('login.php');

// Get stats
$pending_deposits = count(getPendingDeposits());
$total_users = 0;

try {
    $total_users = $pdo->query("SELECT COUNT(*) FROM users")->fetchColumn();
} catch (PDOException $e) {
    error_log("Error counting users: " . $e->getMessage());
}

require_once 'includes/admin_header.php';
?>
<div class="admin-container">
    <?php include 'includes/admin_sidebar.php'; ?>
    <div class="admin-content">
        <h1>Dashboard Overview</h1>
        <div class="stats-grid">
            <div class="stat-card">
                <h3>Total Users</h3>
                <p><?= $total_users ?></p>
            </div>
            <div class="stat-card">
                <h3>Pending Deposits</h3>
                <p><?= $pending_deposits ?></p>
            </div>
        </div>
    </div>
</div>
<?php require_once 'includes/admin_footer.php'; ?>
EOT
);

// 6. Create admin/wallet_approval.php
file_put_contents("$adminFolder/wallet_approval.php", <<<'EOT'
<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/admin_functions.php';

if (!isAdminLoggedIn()) redirect('login.php');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $transaction_id = intval($_POST['transaction_id']);
    $action = sanitizeInput($_POST['action']);
    
    if (in_array($action, ['approve', 'reject'])) {
        if (processWalletTransaction($transaction_id, $action)) {
            $_SESSION['admin_message'] = "Transaction $action" . "d successfully";
        } else {
            $_SESSION['admin_error'] = "Failed to process transaction";
        }
        redirect('wallet_approval.php');
    }
}

$pending_deposits = getPendingDeposits();
require_once 'includes/admin_header.php';
?>
<div class="admin-container">
    <?php include 'includes/admin_sidebar.php'; ?>
    <div class="admin-content">
        <h1>Wallet Approvals</h1>
        <?php if (isset($_SESSION['admin_message'])): ?>
            <div class="alert success"><?= $_SESSION['admin_message'] ?></div>
            <?php unset($_SESSION['admin_message']); ?>
        <?php endif; ?>
        <?php if (isset($_SESSION['admin_error'])): ?>
            <div class="alert error"><?= $_SESSION['admin_error'] ?></div>
            <?php unset($_SESSION['admin_error']); ?>
        <?php endif; ?>
        <?php if (empty($pending_deposits)): ?>
            <p>No pending deposits</p>
        <?php else: ?>
            <table class="data-table">
                <thead>
                    <tr>
                        <th>ID</th><th>User</th><th>Amount</th><th>Date</th><th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($pending_deposits as $deposit): ?>
                    <tr>
                        <td><?= $deposit['id'] ?></td>
                        <td><?= htmlspecialchars($deposit['username']) ?></td>
                        <td>$<?= number_format($deposit['amount'], 2) ?></td>
                        <td><?= date('M d, Y', strtotime($deposit['created_at'])) ?></td>
                        <td>
                            <form method="POST" style="display:inline">
                                <input type="hidden" name="transaction_id" value="<?= $deposit['id'] ?>">
                                <button type="submit" name="action" value="approve" class="btn btn-approve">Approve</button>
                                <button type="submit" name="action" value="reject" class="btn btn-reject">Reject</button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>
<?php require_once 'includes/admin_footer.php'; ?>
EOT
);

// 7. Create admin/logout.php
file_put_contents("$adminFolder/logout.php", <<<'EOT'
<?php
require_once __DIR__ . '/../includes/config.php';
require_once __DIR__ . '/../includes/admin_functions.php';

adminLogout();
redirect('login.php');
EOT
);

// 8. Create admin/includes/admin_header.php
file_put_contents("$adminFolder/includes/admin_header.php", <<<'EOT'
<?php if (!isset($pageTitle)) $pageTitle = "Admin Panel"; ?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= htmlspecialchars($pageTitle) ?> - <?= SITE_NAME ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body>
    <?php if (isAdminLoggedIn()): ?>
    <div class="admin-container">
        <?php include 'admin_sidebar.php'; ?>
        <div class="admin-content">
<?php endif; ?>
EOT
);

// 9. Create admin/includes/admin_footer.php
file_put_contents("$adminFolder/includes/admin_footer.php", <<<'EOT'
        <?php if (isAdminLoggedIn()): ?>
        </div>
    </div>
    <?php endif; ?>
</body>
</html>
EOT
);

// 10. Create admin/includes/admin_sidebar.php
file_put_contents("$adminFolder/includes/admin_sidebar.php", <<<'EOT'
<div class="admin-sidebar">
    <div class="admin-brand">
        <h2><?= SITE_NAME ?></h2>
        <p>Admin Panel</p>
    </div>
    <nav class="admin-nav">
        <a href="index.php" class="<?= basename($_SERVER['PHP_SELF']) === 'index.php' ? 'active' : '' ?>">
            <i class="fas fa-tachometer-alt"></i> Dashboard
        </a>
        <a href="wallet_approval.php" class="<?= basename($_SERVER['PHP_SELF']) === 'wallet_approval.php' ? 'active' : '' ?>">
            <i class="fas fa-wallet"></i> Wallet Approvals
        </a>
        <a href="logout.php">
            <i class="fas fa-sign-out-alt"></i> Logout
        </a>
    </nav>
</div>
EOT
);

// 11. Create assets/css/admin.css
file_put_contents("$assetsFolder/admin.css", <<<'EOT'
:root {
    --primary: #4e73df;
    --primary-dark: #2e59d9;
    --secondary: #f8f9fc;
    --dark: #2c3e50;
    --light: #ffffff;
    --success: #28a745;
    --danger: #dc3545;
}

/* Base Styles */
body {
    font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
    margin: 0;
    padding: 0;
    background-color: var(--secondary);
    color: var(--dark);
}

/* Login Page */
.login-container {
    display: flex;
    justify-content: center;
    align-items: center;
    min-height: 100vh;
}

.login-box {
    background: var(--light);
    padding: 2rem;
    border-radius: 8px;
    box-shadow: 0 0 20px rgba(0,0,0,0.1);
    width: 100%;
    max-width: 400px;
}

.login-box h1 {
    text-align: center;
    margin-bottom: 1.5rem;
}

/* Admin Layout */
.admin-container {
    display: flex;
    min-height: 100vh;
}

.admin-sidebar {
    width: 250px;
    background: var(--dark);
    color: var(--light);
    padding: 1rem 0;
    position: fixed;
    height: 100%;
}

.admin-content {
    flex: 1;
    margin-left: 250px;
    padding: 2rem;
}

/* Tables */
.data-table {
    width: 100%;
    border-collapse: collapse;
    margin-top: 1rem;
    background: var(--light);
}

.data-table th, .data-table td {
    padding: 0.75rem;
    text-align: left;
    border-bottom: 1px solid #eee;
}

.data-table th {
    background-color: var(--primary);
    color: white;
}

/* Buttons */
.btn {
    padding: 0.5rem 1rem;
    border: none;
    border-radius: 4px;
    cursor: pointer;
}

.btn-approve {
    background: var(--success);
    color: white;
}

.btn-reject {
    background: var(--danger);
    color: white;
}

/* Alerts */
.alert {
    padding: 1rem;
    margin-bottom: 1rem;
    border-radius: 4px;
}

.alert.success {
    background: #d4edda;
    color: var(--success);
}

.alert.error {
    background: #f8d7da;
    color: var(--danger);
}

/* Stats Grid */
.stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1rem;
    margin: 1rem 0;
}

.stat-card {
    background: white;
    padding: 1.5rem;
    border-radius: 4px;
    box-shadow: 0 0 5px rgba(0,0,0,0.05);
    text-align: center;
}

/* Responsive */
@media (max-width: 768px) {
    .admin-sidebar {
        width: 100%;
        position: static;
    }
    .admin-content {
        margin-left: 0;
    }
}
EOT
);

echo "<h1>Admin Panel Installed Successfully!</h1>";
echo "<p>All files and folders have been created.</p>";
echo "<p><strong>Next Steps:</strong></p>";
echo "<ol>
    <li>Create the database tables (run the SQL queries)</li>
    <li>Create your first admin user in the admin_users table</li>
    <li>Access the admin panel at: <a href='$adminFolder/login.php'>$adminFolder/login.php</a></li>
    <li><strong style='color:red'>DELETE THIS INSTALLER FILE</strong> for security</li>
</ol>";
echo "<p><strong>Database Tables Needed:</strong></p>";
echo "<pre>
CREATE TABLE admin_users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(50) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    email VARCHAR(100) NOT NULL UNIQUE,
    last_login DATETIME,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    is_active BOOLEAN DEFAULT TRUE
);

CREATE TABLE wallet_transactions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    wallet_id INT NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    transaction_type ENUM('deposit', 'withdrawal') NOT NULL,
    reference VARCHAR(255),
    status ENUM('pending', 'completed', 'rejected') DEFAULT 'pending',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
);
</pre>";